/***************************************************************************
 *
 *  File Name:   doc.c
 *
 *  Description: Functions for opening and closing a Doc file
 *               and decompress/save records as you scroll
 *
 *  Special Thanks to Bill Clagett. Some of this code was adapted from
 *  CSpotRun: Copyright (C) 1998-2000 by Bill Clagett. Also a GPL program.
 *
 *  History:
 *     20 Mar 2000: Created; Jason Sherrill
 *     30 Mar 2000: Added comment headers; Jason Campbell
 *     13 Apr 2000: Created FindRecSizes; Jason Sherrill
 *     14 Apr 2000: Created DecompTxtLen; Jason Sherrill
 *     15 Apr 2000: Changed text buffer from dynamic to storage memory; Jason Sherrill
 *     17 Apr 2000: Created DocGetRecord & DocSaveRecord; Jason Sherrill
 *     18 Apr 2000: Changed back to dynamic & started decomp on demand; Jason Sherrill
 *     19 Apr 2000: Created DocMasterScroll; Jason Sherrill
 *     21 Apr 2000: Created CreateBuffer & RemoveRecFromArr; Jason Sherrill
 *
 ***************************************************************************
 * Copyright (C) 2000 Jason Campbell, Joshua Colvin, Jason Sherrill,
 *					Ben Tobin
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, In., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 *************************************************************************/

#include <PalmOS.h>
#include "doc.h"
#include "convert.h"
#include "docwhoRsc.h"
#include "docview.h"
#include "control.h"


// Private global variables
static DmOpenRef	dbRef;
static MemHandle	rec0Handle;
static DocRec0Type* 	header;
static UInt8		docVersion;

static UInt32		totalNumRecs;		// Including header and bookmarks
static UInt16		numDataRecs;		// Not including header or bookmarks
static UInt16		orgDataRecs;		// Starting number of data recs
static UInt32   	totalTxtLen;		// DocRec0Type length, needs to be tracked
						// or calculated at end for header update
static MemHandle	txtBufHdl;
static MemPtr   	txtBufPtr;
static UInt16   	txtBufLen;

static UInt16   	txtBufLenRem;	
static MemPtr		txtBufPtrNow;

static MemHandle	recSizesHdl;
static UInt16*		recSizesArr;
static UInt16		openRecs[2];


// Private function prototypes
static void FindRecSizes(void);
static UInt16 DecompTxtLen(UInt8* recordPtr, UInt16 recordLen);
static void CreateBuffer(UInt16 size);
// static void DeleteBuffer(void);
static void RemoveRecFromArr(UInt16 indx);
static void UpdateArrEntry(UInt16 rec);


/*************************************************************************
 *
 *  Function Name: DocOpen
 *
 *  Purpose:       To set up the globals for a doc and open the 1st 2 recs
 *
 *  Parameters:    cardNum: The memory card the Doc resides on
 *                 dbID:    The Database ID of the Doc
 *
 *  Return Value:  an unlocked handle to the first 2 records decompressed
 *
 *************************************************************************/

MemHandle DocOpen(UInt16 cardNum, LocalID dbID)
{
	UInt16	bufLen;
	UInt8		numNow;

	// Open Doc database
	dbRef = DmOpenDatabase(cardNum, dbID, dmModeReadWrite);
	ErrNonFatalDisplayIf(!dbRef, "Error: Opening DataBase");
	
	// Open Doc header (record0)
	rec0Handle = DmQueryRecord(dbRef, 0); // &&& DmGetRecord if you want to save changes in DocClose
	ErrNonFatalDisplayIf(!rec0Handle, "Error: Getting Record0");
	header = (DocRec0Type*) MemHandleLock(rec0Handle);
	ErrNonFatalDisplayIf(!header, "Error: Opening Doc Header");
	docVersion = header->version & 0x00FF;  // mask first byte because some docs are wrong

	// Find correct numRecs and decompressed rec sizes
	totalNumRecs = DmNumRecords(dbRef);
	numDataRecs = min(header->numRecs, totalNumRecs - 1); // numRecs-header and bookmarks
	orgDataRecs = numDataRecs;
	FindRecSizes();

	// Create decompress database buffer
	if (numDataRecs > 1)
	{
		numNow = 2;
		openRecs[0] = 1;
		openRecs[1] = 2;
		bufLen = recSizesArr[0] + recSizesArr[1];
	}
	else if (numDataRecs == 1)
	{
		numNow = 1;
		openRecs[0] = 1;
		openRecs[1] = 0;
		bufLen = recSizesArr[0];
	}
	else
	{
		ErrDisplay("Error: numDataRecs = 0 or negative!");
		return NULL;
	}

	CreateBuffer(bufLen);

	DocGetRecords(1, numNow);

	*((Char*)txtBufPtrNow) = '\0';	// Null terminate text field


	MemHandleUnlock(rec0Handle); 
	MemHandleUnlock(txtBufHdl);
	MemHandleUnlock(recSizesHdl);

	return txtBufHdl;
}


/*************************************************************************
 *
 *  Function Name: DocMasterScroll
 *
 *  Purpose:       To control opening/closing recs for decomp on demand
 *
 *  Parameters:    dir:      Scroll direction
 *                 isEdited: Has the textfield been edited
 *                 changed:  returns how much text has changed for scroll adjust
 *                 s:        returns true if you should keep the same buffer
 *
 *  Return Value:  an unlocked handle to the next 2 recs in a text buffer
 *
 *************************************************************************/

MemHandle DocMasterScroll(WinDirectionType dir, Boolean isEdited, UInt16* changed, Boolean* s)
{
	MemHandle	oldBufHdl;
	MemPtr	oldBufPtr;
	MemPtr	startNxtPtr;
	UInt16	newBufLen;

	UInt16	nextRec;
	UInt16	newLen;
	UInt16	remLen;

	FormPtr   frmPtr;
	FieldPtr  fldPtr;

	if (numDataRecs == 1)  // Small Doc, 1 record, keep same buffer
	{
		// &&& check under 4K, yes: NULL, no: create new record... On Close only
		*s = true;
		*changed = 0;
		return NULL;
	}

	if (dir == winUp)
	{
		if (openRecs[0] <= 1)
		{
			*s = true;
			*changed = 0;
			return NULL;  // At top already, keep same buffer
		}
	}
	else  //(dir == winDown)
	{
		if (openRecs[1] >= numDataRecs)
		{
			*s = true;
			*changed = 0;
			return NULL;  // At bottom already, keep same buffer
		}
	}

	recSizesArr = (UInt16*) MemHandleLock(recSizesHdl);
	ErrNonFatalDisplayIf(!recSizesArr, "Error: Locking recSizes Array");

	oldBufHdl = txtBufHdl;

	if (isEdited)
	{
		frmPtr = FrmGetActiveForm();
		fldPtr = ControlGetPtrByID(frmPtr, DocField);
		newLen = FldGetTextLength(fldPtr);
		oldBufPtr = MemHandleLock(oldBufHdl);

		if (dir == winUp)
		{
			nextRec = openRecs[0] - 1;

			if (newLen <= 4096)       // 4K, compress into 1 rec, delete 2nd
			{
				DocSaveRecord(openRecs[0], oldBufPtr, newLen);

				UpdateArrEntry(openRecs[0]);

				DmRemoveRecord(dbRef, openRecs[1]);
				RemoveRecFromArr(openRecs[1]-1);
				numDataRecs -= 1;

				newBufLen = recSizesArr[nextRec-1] + recSizesArr[openRecs[0]-1];

				CreateBuffer(newBufLen);

				DocGetRecords(nextRec, 2);

				DocViewSetClean();

				*changed = 0;
				openRecs[0] = nextRec;
				openRecs[1] = nextRec + 1;
			}
			else
			{
				startNxtPtr = oldBufPtr + (newLen - 4096);

				DocSaveRecord(openRecs[1], startNxtPtr, 4096); // &&& 4096-1?

				UpdateArrEntry(openRecs[1]);

				remLen = newLen - 4096;
				newBufLen = recSizesArr[nextRec-1] + remLen;

				CreateBuffer(newBufLen);

				// Get next record
				DocGetRecords(nextRec, 1);

				if (remLen <= 4096)
				{
					DocSaveRecord(openRecs[0], oldBufPtr, remLen);

					UpdateArrEntry(openRecs[0]);

					DocGetRecords(openRecs[0], 1);

					DocViewSetClean();
				}
				else  // too big, set next buffer dirty until 2 recs fit within 8K
				{
					MemMove(txtBufPtrNow, oldBufPtr, remLen);
					txtBufPtrNow += remLen;
					if (remLen < txtBufLenRem)
						txtBufLenRem -= remLen;
					else
						txtBufLenRem = 0;

					DocViewSetDirty();
				}

				*changed = recSizesArr[nextRec-1];
				openRecs[0] = nextRec;
				openRecs[1] = nextRec + 1;
			}
		}
		else  // down
		{
			nextRec = openRecs[1] + 1;

			if (newLen <= 4096)       // 4K, compress into 1 rec, delete 2nd
			{
				DocSaveRecord(openRecs[0], oldBufPtr, newLen);

				UpdateArrEntry(openRecs[0]);

				DmRemoveRecord(dbRef, openRecs[1]);
				RemoveRecFromArr(openRecs[1]-1);
				numDataRecs -= 1;

				newBufLen = recSizesArr[openRecs[0]-1] + recSizesArr[openRecs[1]-1];

				CreateBuffer(newBufLen);

				DocGetRecords(openRecs[0], 2);

				DocViewSetClean();

				*changed = 0;
				// openRecs[0] = openRecs[0]; // Both stay the same
				// openRecs[1] = nextRec - 1;
			}
			else
			{
				startNxtPtr = oldBufPtr + 4096;

				DocSaveRecord(openRecs[0], oldBufPtr, 4096);

				UpdateArrEntry(openRecs[0]);

				remLen = newLen - 4096;
				newBufLen = remLen + recSizesArr[nextRec-1];

				CreateBuffer(newBufLen);

				if (remLen <= 4096)
				{
					DocSaveRecord(openRecs[1], startNxtPtr, remLen);

					UpdateArrEntry(openRecs[1]);

					DocGetRecords(openRecs[1], 1);

					DocViewSetClean();
				}
				else  // too big, set next buffer dirty until 2 recs fit within 8K
				{
					MemMove(txtBufPtrNow, startNxtPtr, remLen);
					txtBufPtrNow += remLen;
					if (remLen < txtBufLenRem)
						txtBufLenRem -= remLen;
					else
						txtBufLenRem = 0;

					DocViewSetDirty();
				}

				// Get next record
				DocGetRecords(nextRec, 1);

				*changed = 4096;
				openRecs[0] = nextRec - 1;
				openRecs[1] = nextRec;
			}
		}
		MemHandleUnlock(oldBufHdl);
	}
	else	// hasn't been edited, just scroll
	{
		if (dir == winUp)
		{
			nextRec = openRecs[0] - 1;
			newBufLen = recSizesArr[nextRec-1] + recSizesArr[openRecs[0]-1];

			CreateBuffer(newBufLen);

			DocGetRecords(nextRec, 2);

			*changed = recSizesArr[nextRec-1];
			openRecs[0] = nextRec;
			openRecs[1] = nextRec + 1;
		}
		else  // down
		{
			nextRec = openRecs[1] + 1;
			newBufLen = recSizesArr[openRecs[1]-1] + recSizesArr[nextRec-1];

			CreateBuffer(newBufLen);

			DocGetRecords(openRecs[1], 2);

			*changed = recSizesArr[openRecs[0]-1];
			openRecs[0] = nextRec - 1;
			openRecs[1] = nextRec;
		}
	}
	*s = false;
//	if (oldBufHdl)
//		MemHandleFree(oldBufHdl);

	*((Char*)txtBufPtrNow) = '\0';	// Null terminate text field

	MemHandleUnlock(txtBufHdl);	// Unlock from lock in CreateBuffer()
	MemHandleUnlock(recSizesHdl);

	return txtBufHdl;
}


/*************************************************************************
 *
 *  Function Name: DocGetRecords
 *
 *  Purpose:       Get the specified recs and put them into txtBuf
 *
 *  Parameters:    startRec: The first record to get
 *                 numRecs:  The number of records to get
 *
 *  Return Value:  nothing, could return a handle for other uses (see below)
 *
 *************************************************************************/

void DocGetRecords(UInt16 startRec, UInt8 numRecs)
{
	MemHandle	curRecHdl;
	MemPtr	curRecPtr;
	UInt16	curRecLen;
	UInt8 	curRecNum;
	UInt8 	i;

	UInt16	decLen;
	UInt16	maxDecLen;

	decLen = 0;

	// Iterate through data records for decompression
	for (i = 0; i < numRecs && txtBufLenRem > 0; ++i)
	{
		curRecNum = startRec + i;
		if (curRecNum > numDataRecs)
		{
			ErrDisplay("Error: curRecNum > numDataRecs in DocGetRecords");
			return;
		}
		curRecHdl = DmQueryRecord(dbRef, curRecNum);
		ErrNonFatalDisplayIf(!curRecHdl, "Error: Getting curRecHdl in DocGetRecords");
		curRecPtr = MemHandleLock(curRecHdl);
		ErrNonFatalDisplayIf(!curRecPtr, "Error: Locking curRecPtr in DocGetRecords");
		curRecLen = MemHandleSize(curRecHdl);

		// Check compression type, decompress or copy if text
		switch (docVersion)
		{
			case 2:

				maxDecLen = min(txtBufLenRem, recSizesArr[curRecNum-1]);

				decLen = Decompress(curRecPtr, curRecLen, txtBufPtrNow, maxDecLen);

				txtBufPtrNow += decLen;
				if (decLen < txtBufLenRem)
					txtBufLenRem -= decLen;
				else
					txtBufLenRem = 0;
				break;

			case 1:

				decLen = min(txtBufLenRem, curRecLen);

				MemMove(txtBufPtrNow, curRecPtr, decLen);

				txtBufPtrNow += decLen;
				if (decLen < txtBufLenRem)
					txtBufLenRem -= decLen;
				else
					txtBufLenRem = 0;

				break;

			default:

				ErrDisplay("Error: Unknown Compression Version");
		}

		MemHandleUnlock(curRecHdl);
	}

//	could be used for searching & jump to line numbers, bookmarks too :)
//	return txtBufHdl; 
}


/*************************************************************************
 *
 *  Function Name: DocSaveRecord
 *
 *  Purpose:       Save a decompressed record into the specified recNum
 *
 *  Parameters:    recNum:    The record number to save to
 *                 startAddr: The beginning of the text in a buffer
 *                 decLen:    The length of text to compress and save
 *
 *  Return Value:  size of the resulting compressed text
 *
 *************************************************************************/

UInt16 DocSaveRecord(UInt16 recNum, MemPtr startAddr, UInt16 decLen)
{
	MemHandle	newRecHdl;
	MemPtr   	newRecPtr;
	MemHandle	oldRecHdl;

	UInt16	maxLen;
	UInt16	compLen;

	maxLen = min(4096, decLen);

	oldRecHdl = DmQueryRecord(dbRef, recNum);

	newRecHdl = DmNewHandle(dbRef, maxLen);
	ErrNonFatalDisplayIf(!newRecHdl, "Error: Creating newRecHdl db record in Save");
	newRecPtr = MemHandleLock(newRecHdl);
	ErrNonFatalDisplayIf(!newRecPtr, "Error: Locking newRecPtr in Save");

// &&& deal with uncompressed Docs, docVersion == 1
	compLen = Compress(startAddr, maxLen, newRecPtr, maxLen);

	MemHandleUnlock(newRecHdl);
	DmAttachRecord(dbRef, &recNum, newRecHdl, &oldRecHdl);

	newRecHdl = DmResizeRecord(dbRef, recNum, compLen);

	if (oldRecHdl)
		MemHandleFree(oldRecHdl);	// &&& dmDeleteRecord, already detached though

	return compLen;
}


/*************************************************************************
 *
 *	Function Name: DocClose
 *
 *	Purpose:       Close a Doc, save open records, update Doc header
 *
 *	Parameters:    isEdited: Have the 2 open records been edited?
 *
 *	Return Value:  nothing
 *
 *************************************************************************/

void DocClose(Boolean isEdited)
{
	FormPtr  frmPtr;
	FieldPtr fldPtr;
	MemPtr   startNxtPtr;

	UInt16   newLen;
	UInt16   remLen;
//	Int16    recChange;
//	UInt16   newNumRecs;

	if (dbRef != NULL)
	{
		header = (DocRec0Type*) MemHandleLock(rec0Handle);

		// Save 2 open recs, add/remove recs as necessary, update header
		if (isEdited)
		{
			frmPtr = FrmGetActiveForm();
			fldPtr = ControlGetPtrByID(frmPtr, DocField);
			newLen = FldGetTextLength(fldPtr);
			txtBufPtr = MemHandleLock(txtBufHdl);

			if (newLen <= 4096)       // 4K, compress into 1 rec, delete 2nd
			{
				DocSaveRecord(openRecs[0], txtBufPtr, newLen);

				DmRemoveRecord(dbRef, openRecs[1]);
				numDataRecs -= 1;
			}
			else
			{
				startNxtPtr = txtBufPtr + 4096;

				DocSaveRecord(openRecs[0], txtBufPtr, 4096);

				remLen = newLen - 4096;

				if (remLen <= 4096)
				{
					DocSaveRecord(openRecs[1], startNxtPtr, remLen);
				}

				// too big, carry text down, adjusting rec sizes to 4K
				else  //  until it fits, or add rec(s) at the end
				{
/* &&&
					newBufLen = remLen + recSizesArr[nextRec-1];

					CreateBuffer(newBufLen);

					MemMove(txtBufPtrNow, startNxtPtr, remLen);
					txtBufPtrNow += remLen;
					if (remLen < txtBufLenRem)
						txtBufLenRem -= remLen;
					else
						txtBufLenRem = 0;

					// Get next record
					DocGetRecords(nextRec, 1);

					openRecs[0] = nextRec - 1;
					openRecs[1] = nextRec;
*/
				}
			}
		}

		// correct the number of data recs in the Doc header
/*		if (orgDataRecs != numDataRecs)
		{
			recChange = numDataRecs - orgDataRecs;
			newNumRecs = header->numRecs + recChange;
			// numRecs: 2 byte field at offsets 8 & 9
			DmWrite(header, 8, &newNumRecs, 2);
		}
	
		// &&& correct total text length: header->length
*/

		// Close record0
		MemHandleUnlock(rec0Handle);
		rec0Handle = NULL;
		header = NULL;

		// Free record sizes array
		MemHandleFree(recSizesHdl);
		recSizesHdl = NULL;
		recSizesArr = NULL;

		// Free text buffer (decompress buffer)
//		DeleteBuffer(); Handled by system when it closes Doc View Form

		// Close Doc Database
		DmCloseDatabase(dbRef);
		dbRef = NULL;
	}
}


/*************************************************************************
 *
 *	Function Name: FindRecSizes
 *
 *	Purpose:       Creates and populates recSizesArr
 *
 *	Parameters:    nothing
 *
 *	Return Value:  nothing
 *
 *************************************************************************/

static void FindRecSizes(void)
{
	MemHandle	curRecHdl;
	MemPtr	curRecPtr;
	UInt16	curRecLen;
	UInt16	decTxtLen;		// decompressed text length of current record
	UInt16	recIndex;

	totalTxtLen = 0;

	// Create record sizes array
	recSizesHdl = MemHandleNew(numDataRecs*sizeof(*recSizesArr));
	ErrNonFatalDisplayIf(!recSizesHdl, "Error: Creating recSizes Array Handle");
	recSizesArr = (UInt16*) MemHandleLock(recSizesHdl);
	ErrNonFatalDisplayIf(!recSizesArr, "Error: Locking recSizes Array");

	// Iterate through records, find decomp length of each record, keep a total
	for (recIndex=1; recIndex <= numDataRecs; recIndex++)
	{
		curRecHdl = DmQueryRecord(dbRef, recIndex);
		ErrNonFatalDisplayIf(!curRecHdl, "Error: Getting curRecHdl in FindRecSizes");
		curRecPtr = MemHandleLock(curRecHdl);
		ErrNonFatalDisplayIf(!curRecPtr, "Error: Locking curRecPtr in FindRecSizes");
		curRecLen = MemHandleSize(curRecHdl);

		if (curRecPtr)
		{
			switch (docVersion)
			{
				case 2:
					decTxtLen = DecompTxtLen(curRecPtr, curRecLen);
					recSizesArr[recIndex-1] = decTxtLen;
					totalTxtLen += decTxtLen;
					break;
				case 1:
					recSizesArr[recIndex-1] = curRecLen;
					totalTxtLen += curRecLen;
					break;
				default:
					ErrDisplay("Error: Unknown Compression Version");
			}
		}
		MemHandleUnlock(curRecHdl);
	}
}


/*************************************************************************
 *
 *  Function Name: DecompTxtLen
 *
 *  Purpose:       Finds the decompressed length of a record
 *
 *  Parameters:    recordPtr: Points to the record to size
 *                 recordLen: The compressed length of the record
 *
 *  Return Value:  the decompressed length
 *
 *************************************************************************/

static UInt16 DecompTxtLen(UInt8* recordPtr, UInt16 recordLen)
{
	UInt16 c;
	UInt16 decompLen;
	UInt8* endAddr;

	decompLen = 0;
	endAddr = &recordPtr[recordLen];

	while (recordPtr < endAddr)
	{
		// Take a byte from the input buffer
		c = *(recordPtr++);

		// Type A command, copy next c bytes directly
		if ( c >= 1 && c <= 8 )
		{
			decompLen += c;
			if (recordPtr + c >= endAddr)
				break;	// don't want to access past record end
			recordPtr += c;
		}

		// 0,9...7F represent themselves, copy that byte directly
		else if ( c <= 0x7F )
			decompLen++;

		// Type B command, sliding window sequence
		else if ( c >= 0x80 && c <= 0xBF )
		{
			if (recordPtr >= endAddr)
				break;	// don't want to access past record end
			c = (c << 8) | *(recordPtr++);
			decompLen += (c & 0x0007) + 3;
		}
		
		// Type C command, copy a space + char formed by lower 7 bits
		else if (c >= 0xC0)
			decompLen += 2;

		else
			ErrDisplay("Error: Encountered strange byte in DecompTxtLen");
	}

	return decompLen;
}


/*************************************************************************
 *
 *	Function Name: CreateBuffer
 *
 *	Purpose:       Creates a text buffer of specified size -> txtBufHdl
 *
 *	Parameters:    size: The byte length of the needed buffer
 *
 *	Return Value:  nothing
 *
 *************************************************************************/

// Must unlock txtBufHdl after use
static void CreateBuffer(UInt16 size)
{
	UInt32 chunkMax;
	UInt32 freeBytes;

	MemHeapFreeBytes (0, &freeBytes, &chunkMax);
	txtBufLen = 0;

	if (size < chunkMax)
	{
		txtBufLen = size;
		txtBufHdl = MemHandleNew(txtBufLen+1);	// for Null terminating
		ErrNonFatalDisplayIf(!txtBufHdl, "Error: Creating New txtBufHdl");
		txtBufPtr = MemHandleLock(txtBufHdl);
		ErrNonFatalDisplayIf(!txtBufPtr, "Error: Locking txtBufPtr");
		txtBufPtrNow = txtBufPtr;
		txtBufLenRem = txtBufLen;
	}
	else
		ErrDisplay("Error: not enough dynamic memory to create text buffer");
}

/*
static void DeleteBuffer(void)
{
	if (txtBufHdl)
	{
		MemHandleFree(txtBufHdl);
		txtBufHdl = NULL;
		txtBufPtr = NULL;
		txtBufLen = 0;
		txtBufPtrNow = NULL;
		txtBufLenRem = 0;
	}
	else
		ErrDisplay("Error: txtBufHdl doesn't exist for DeleteBuffer");
}
*/

/*************************************************************************
 *
 *	Function Name: RemoveRecFromArr
 *
 *	Purpose:       Removes a record entry from recSizesArr
 *
 *	Parameters:    indx: Array index to remove (recNum - 1)
 *
 *	Return Value:  nothing
 *
 *************************************************************************/

static void RemoveRecFromArr(UInt16 indx)
{
	UInt16 i;

	recSizesArr = (UInt16*) MemHandleLock(recSizesHdl);
	ErrNonFatalDisplayIf(!recSizesArr, "Error: Locking recSizes Array");

	if (indx <= numDataRecs-1)
	{
		for (i=indx; i < numDataRecs-2; ++i)
		{
			recSizesArr[i] = recSizesArr[i+1];
		}
		recSizesArr[i] = 0;  // last record
//		MemHandleResize(recSizesHdl, numDataRecs-1);
	}
	else
		ErrDisplay("Error: rec Entry to remove from recSizesArr past bounds");

	MemHandleUnlock(recSizesHdl);
}


/*************************************************************************
 *
 *	Function Name: UpdateArrEntry
 *
 *	Purpose:       Finds the new decomp len of a rec entry in recSizesArr
 *
 *	Parameters:    indx: Array index to remove (recNum - 1)
 *
 *	Return Value:  nothing
 *
 *************************************************************************/

static void UpdateArrEntry(UInt16 rec)
{
	MemHandle	recHdl;
	MemPtr	recPtr;
	UInt16	recLen;

	recSizesArr = (UInt16*) MemHandleLock(recSizesHdl);
	ErrNonFatalDisplayIf(!recSizesArr, "Error: Locking recSizes Array");

	if (rec <= numDataRecs)
	{
		recHdl = DmQueryRecord(dbRef, rec);
		ErrNonFatalDisplayIf(!recHdl, "Error: Getting recHdl");
		recPtr = MemHandleLock(recHdl);
		ErrNonFatalDisplayIf(!recPtr, "Error: Locking recPtr");
		recLen = MemHandleSize(recHdl);

		recSizesArr[rec-1] = DecompTxtLen(recPtr, recLen);
		MemHandleUnlock(recHdl);
	}
	else
		ErrDisplay("Error: rec Entry to update in recSizesArr past bounds");

	MemHandleUnlock(recSizesHdl);
}


