/*************************************************************************
 *
 *    File Name: doclist.c
 *
 *    Description: Functions relating to the Doc List form
 *
 *    History:
 *       20 Mar 2000: Created; Joshua Colvin
 *       30 Mar 2000: Added comment headers; Jason Campbell
 *       01 Apr 2000: Filled in comment headers; Joshua Colvin
 *       01 Apr 2000: Created DocListPushButtonType; Joshua Colvin
 *       05 Apr 2000: Created DocListHandleControlSelect; Joshua Colvin
 *       05 Apr 2000: Created DocListCustomTableSaveData; Joshua Colvin
 *       05 Apr 2000: Uncommented call to DocViewSetDocInfo; Jason Sherrill
 *       05 Apr 2000: Included docview.h; Jason Sherrill
 *       13 Apr 2000: Changed "Category" button to "Details"; Joshua Colvin
 *       14 Apr 2000: Added DocListHandleCategorySelect; Joshua Colvin
 *       14 Apr 2000: Completed category support for display; Joshua Colvin
 *       20 Apr 2000: Added DocListScroll; Joshua Colvin
 *       24 Apr 2000: Added DocListUpdateScrollers; JoshuaColvin
 *       24 Apr 2000: Added DocListHandleControlRepeat; Joshua Colvin
 *       27 Apr 2000: Added DocListBeamCategory; Joshua Colvin
 *
 *************************************************************************
 * Copyright (C) 2000 Jason Campbell, Joshua Colvin, Jason Sherrill,
 *                    Ben Tobin
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, In., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 *************************************************************************/

#include <PalmOS.h>
#include "doclist.h"
#include "docwhoRsc.h"
#include "control.h"
#include "doclistdb.h"
#include "docview.h"
#include "docdetails.h"
#include "beam.h"
#include "version.h"
#include "menu.h"

// Private function prototypes
static void DocListFormInit(FormPtr frmP);
static void DocListFormClose(void);
static UInt16 DocListGetRecordIndex(EventPtr eventP);
static Err DocListCustomTableLoadData(void *tableP, Int16 row, Int16 column,
                  Boolean editable, MemHandle *dataHandleP, Int16 *dataOffsetP,
                  Int16 *dataSizeP, FieldPtr fldP);
static void DocListLoadTable(TablePtr tableP);
static Boolean DocListCustomTableSaveData(void *tableP, Int16 row, Int16 column);
static void DocListInitializeTable(FormPtr frmP);
static Boolean DocListHandleTableSelection(EventPtr eventP);
static Boolean DocListHandleControlSelect(EventPtr eventP);
static void DocListHandleCategorySelect(FormPtr frmP);
static void DocListScroll(FormPtr frmP, Int32 rows);
static void DocListUpdateScrollers(FormPtr frmP,
                                   UInt16 lastVisibleRecordIndex);
static Boolean DocListHandleControlRepeat(EventPtr eventP);

// Application-wide global variable declarations
// gCurrentDocIndex contains the location in DocListDB of
//  the information about the current Doc file open
//  (should be declared extern in other files that use it)
UInt16 gCurrentDocIndex;

// Local global variable declarations
// gSelectedPushButton is used to keep track of which pushbutton
// is selected (stored as resource ID)
static UInt16 gSelectedPushButton;
// gTopVisibleRecord is used by DocListLoadTable to keep track of which
//  record to start the display at
static Int32 gTopVisibleRecordIndex;
// gCurrentCategory and gCurrentCategoryName store info about current category
static UInt16 gCurrentCategory;
static Char gCurrentCategoryName[dmCategoryLength];

/*************************************************************************
 *
 *    Function Name: DocListFormInit
 *
 *    Purpose:       Initialize global variables, initialize DocListDB and
 *                    set default pushbutton selection
 *
 *    Parameters:    frmP: pointer to DocListForm
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
static void DocListFormInit(FormPtr frmP)
{
	ControlPtr triggerP;
	ControlPtr beamButtonP;
	ControlPtr openButtonP;
	Boolean result;
	
	// Initialize global variables
	gTopVisibleRecordIndex = 0;
	gCurrentCategory = dmAllCategories;
	gCurrentDocIndex = 0;
	
	// Turn off "Beam" button if OS < 3.0
	result = VersionCheckOS(RomVer3_0);
	if (false == result)
	{
		beamButtonP = ControlGetPtrByID(frmP, DocListBeamButton);
		CtlHideControl(beamButtonP);
	}
	
	// Select the "Open" button as default
	FrmSetControlGroupSelection(frmP, DocListPushButtonGroup,
	                            DocListOpenButton);
	openButtonP = ControlGetPtrByID(frmP, DocListOpenButton);
	CtlHitControl(openButtonP);

	// Initialize DocListDB
	// DocListDBInit moved to palmread.c	
	DocListDBPopulate();
	DocListInitializeTable(frmP);
	
	// Draw category stuff
	DocListDBCategoryGetName(gCurrentCategory, gCurrentCategoryName);
	triggerP = ControlGetPtrByID(frmP, DocListCategoryPopup);
	CategorySetTriggerLabel(triggerP, gCurrentCategoryName);
	
}

/*************************************************************************
 *
 *    Function Name: DocListFormClose
 *
 *    Purpose:       Close database used by DocListForm
 *
 *    Parameters:    none
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
static void DocListFormClose(void)
{
	// DocListDBClose has been moved to palmread.c
}

/*************************************************************************
 *
 *    Function Name: DocListBeamCategory
 *
 *    Purpose:       Beam all Doc files in current category
 *
 *    Parameters:    none
 *
 *    Return Value:  non-zero if error occurs
 *
 *
 *************************************************************************/
Err DocListBeamCategory(void)
{
	Err err;
	
	err = BeamCategory(gCurrentCategory);
	
	return err;
}

/*************************************************************************
 *
 *    Function Name: DocListGetRecordIndex
 *
 *    Purpose:       return the recordIndex of the item selected in form
 *
 *    Parameters:    eventP: pointer to event sent to event handler
 *
 *    Return Value:  The index of the record selected by user
 *
 *
 *************************************************************************/
static UInt16 DocListGetRecordIndex(EventPtr eventP)
{
	UInt16 recordIndex;
	
	recordIndex = TblGetRowID(eventP->data.tblSelect.pTable,
	                            eventP->data.tblSelect.row);
	                            
	return recordIndex;
}

/*************************************************************************
 *
 *    Function Name: DocListCustomTableLoadData
 *
 *    Purpose:       Called by table handler to load data into each table
 *                    cell
 *                    (Load handle to name of Doc file)
 *                    
 *
 *    Parameters:    In:
 *                   tableP:      Pointer to a table object
 *                   row:         Row number of the table item to load
 *                   column:      Column number of the table item to load
 *                   editable:    If table is being edited
 *
 *                   Out:
 *                   dataHandleP: Unlocked handle of a block containing a
 *                                 null-terminated string
 *                   dataOffsetP: Offset from start of black to start of
 *                                 of the text string
 *                   dataSizeP:   Allocated size of text string
 *                   fldP:        Pointer to the text field in this table
 *                                 cell
 *
 *    Return Value: 0 upon success or an error if unsuccessful
 *
 *
 *************************************************************************/
static Err DocListCustomTableLoadData(void *tableP, Int16 row, Int16 column,
                  Boolean editable, MemHandle *dataHandleP, Int16 *dataOffsetP,
                  Int16 *dataSizeP, FieldPtr fldP)
{
	Int16 recordIndex;
	FieldAttrType attr;

	//Get the record number that corresponds to the table item to draw
	recordIndex = TblGetRowID(tableP, row);
	
	// Set handle directly to Doc name in DocListDB
	DocListDBGetNameHandle(recordIndex, dataHandleP, dataOffsetP, dataSizeP);
	
	// Get field attributes
	FldGetAttributes(fldP, &attr);
	
	// Turn off underline mode
	attr.underlined = noUnderline;
	
	// Set new field attributes
	FldSetAttributes(fldP, &attr);
	
	// Set maximum characters allowed
	FldSetMaxChars( fldP, (dmDBNameLength - 1) );
	
	return 0;
}

/*************************************************************************
 *
 *    Function Name: DocListCustomTableSaveData
 *
 *    Purpose:       Called by table handler to save edited data from
 *                    each table cell
 *                    (Update the name of the Doc file)
 *                    
 *
 *    Parameters:    tableP:      Pointer to a table object
 *                   row:         Row number of the table item to save
 *                   column:      Column number of the table item to save
 *
 *    Return Value: true if table needs to be redrawn, false if not
 *
 *
 *************************************************************************/
static Boolean DocListCustomTableSaveData(void *tableP, Int16 row, Int16 column)
{
	UInt16 recordIndex;
	FieldPtr fldP;
	Boolean redraw;
	
	redraw = false;
	
	// Get the current field which contains the edited Doc name
	fldP = TblGetCurrentField(tableP);
	
	// Update Doc name to match new name if it has changed
	if ( FldDirty(fldP) )
	{
		//Get the record number that corresponds to the table item to draw
		recordIndex = TblGetRowID(tableP, row);
		
		redraw = DocListDBUpdateDocName(recordIndex);
	}
	
	return redraw;
}

/*************************************************************************
 *
 *    Function Name: DocListLoadTable
 *
 *    Purpose:       Assigns table rows to record entries
 *                    When scrolling: update gTopVisibleRecordIndex and call
 *                                    this function
 *                    Category changes: Call this function
 *
 *    Parameters:    tableP: pointer to table structure
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
static void DocListLoadTable(TablePtr tableP)
{
	Int16 rowIndex;
	Int16 numRows;
	MemHandle recordH;
	Int16 recordIndex;
	Int16 bottomVisibleRecordIndex;
	
	recordIndex = gTopVisibleRecordIndex;
	bottomVisibleRecordIndex = 0;
	
	// For each row in the table, store the record number in the
	//  table item that will display the record
	numRows = TblGetNumberOfRows(tableP);
 	for (rowIndex = 0; rowIndex < numRows; rowIndex++, recordIndex++)
	{
		recordH = DocListDBQueryNextInCategory(&recordIndex,
		                                       gCurrentCategory);
		
		
		if (recordH)
		{
			// Set style to for displaying text
			TblSetItemStyle(tableP, rowIndex, 0, textTableItem);

			// Record was found so store record number in the table rowID
			TblSetRowID(tableP, rowIndex, recordIndex);
			TblSetRowUsable(tableP, rowIndex, true);
			bottomVisibleRecordIndex = recordIndex;
		}
		else // NULL == recordH; record was not found
		{
			// Set any extra table rows as unusable
			TblSetRowUsable(tableP, rowIndex, false);
		}
		// Mark the row invalid so that it will draw the new contents
		TblMarkRowInvalid(tableP, rowIndex);
	}
	
	// Redraw table
	// ####
	//TblRedrawTable(tableP);
	
	// Update the scroll arrows
	DocListUpdateScrollers(FrmGetActiveForm(), bottomVisibleRecordIndex);
}

/*************************************************************************
 *
 *    Function Name: DocListInitializeTable
 *
 *    Purpose:       Setup table to display Doc information
 *
 *    Parameters:    frmP: pointer to DocListForm
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
static void DocListInitializeTable(FormPtr frmP)
{
	TablePtr tableP;
	
	// Get the table
	tableP = ControlGetPtrByID(frmP, DocListTable);
	
	// Load the records into the table
	DocListLoadTable(tableP);
	
	// Set the column usable so that it draws and accepts user input
	TblSetColumnUsable(tableP, 0, true);
	
	// Set LoadData to initialize the Doc names in the table
	TblSetLoadDataProcedure(tableP, 0, DocListCustomTableLoadData);
	
	// Set SaveData to save any edited Doc names in the table
	TblSetSaveDataProcedure(tableP, 0, DocListCustomTableSaveData);
}

/*************************************************************************
 *
 *    Function Name: DocListUpdateScrollers
 *
 *    Purpose:       Updates the Scroller buttons
 *
 *    Parameters:    eventP: pointer to event
 *
 *    Return Value:  true if event handled
 *
 *
 *************************************************************************/
static void DocListUpdateScrollers(FormPtr frmP,
                                   UInt16 lastVisibleRecordIndex)
{
	UInt16 upIndex;
	UInt16 downIndex;
	Boolean scrollableUp;
	Boolean scrollableDown;
	UInt16 recordIndex;
	Err err;
	
	// Find out if records exist above displayed records
	recordIndex = gTopVisibleRecordIndex;
	err = DocListDBSeekRecordInCategory(&recordIndex, 1, dmSeekBackward,
	                                    gCurrentCategory);
	// If there was no error, records exist above top visible record
	if (err)
	{
		scrollableUp = false;
	}
	else
	{
		scrollableUp = true;
	}
	
	// Find out if records exist below displayed records
	recordIndex = lastVisibleRecordIndex;
	err = DocListDBSeekRecordInCategory(&lastVisibleRecordIndex, 1,
	                                    dmSeekForward, gCurrentCategory);
	// If there was no error, records exist below last visible record
	if (err)
	{
		scrollableDown = false;
	}
	else
	{
		scrollableDown = true;
	}
	
	upIndex   = FrmGetObjectIndex(frmP, DocListScrollUpButton);
	downIndex = FrmGetObjectIndex(frmP, DocListScrollDownButton);

	FrmUpdateScrollers(frmP, upIndex, downIndex, scrollableUp,
	                   scrollableDown);
}

/*************************************************************************
 *
 *    Function Name: DocListScroll
 *
 *    Purpose:       Scroll list requested number of rows
 *
 *    Parameters:    rows: Number of rows to scroll
 *                    down if positive, up if negative
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
static void DocListScroll(FormPtr frmP, Int32 rows)
{
	TablePtr tableP;
	
	tableP = ControlGetPtrByID(frmP, DocListTable);
	
	// Update top record of table
	gTopVisibleRecordIndex += rows;
	
	if (gTopVisibleRecordIndex < 0)
	{
		gTopVisibleRecordIndex = 0;
	}
	
	// Force update of table
	FrmUpdateForm(DocListForm, frmUpdateEventReloadTable);
}

/*************************************************************************
 *
 *    Function Name: DocListHandleTableSelection
 *
 *    Purpose:       Handle table item selection event
 *                    calls TblReleaseFocus if necessary
 *
 *    Parameters:    eventP: pointer to event
 *
 *    Return Value:  true if event handled
 *
 *
 *************************************************************************/
static Boolean DocListHandleTableSelection(EventPtr eventP)
{
	FormPtr frmP;
	UInt16 cardNumber;
	LocalID dbID;
	Char name[dmDBNameLength];
	Boolean handled;
	UInt16 selection;
	
	// Get record index of item selected
	gCurrentDocIndex = DocListGetRecordIndex(eventP);
	
	// Get info on selected Doc file
	DocListDBGetInfo(gCurrentDocIndex, &cardNumber, &dbID,
	                 name, NULL, NULL, NULL, NULL);

	// Get pointer to current form
	frmP = FrmGetActiveForm();
	
	// Do appropriate action on selected item
	handled = false;
	switch (gSelectedPushButton)
	{
		case DocListOpenButton:
			// Release focus or item will still be highlighted
			FrmSetFocus(frmP, noFocus);
			// Open Doc file
			DocViewSetDocInfo(cardNumber, dbID);
			FrmGotoForm(DocViewForm); //####Open selection
			handled = true;
			break;
		case DocListDetailsButton:
			// Release focus or item will still be highlighted
			FrmSetFocus(frmP, noFocus);
			//FrmCustomAlert(DebugAlert, "Details not implemented yet!", NULL, NULL);
			FrmPopupForm(DocDetailsForm);
			// DB and Table updated in DocDetailsForm
			handled = true;
			break;
		case DocListRenameButton:
			// Rename is handled elsewhere
			handled = true;
			break;
		case DocListDeleteButton:
			// Ask user to confirm delete
			selection = FrmCustomAlert(ConfirmAlert, "Delete", name, NULL);
			if(0 == selection) // User selected "OK"
			{
				// Delete Doc file
				DmDeleteDatabase(cardNumber, dbID);

				// Force update of DB and table
				FrmUpdateForm(DocListForm, frmUpdateEventReloadDBAndTable);
			}
			// Release focus or item will still be highlighted
			FrmSetFocus(frmP, noFocus);
			handled = true;
			break;
		case DocListBeamButton:
			// Release focus or item will still be highlighted
			FrmSetFocus(frmP, noFocus);
		   StrCat(name, " (Doc File)");
			BeamDatabase(cardNumber, dbID, "doc.pdb", name);
			handled = true;
			break;
	}
	return handled;
}

/*************************************************************************
 *
 *    Function Name: DocListHandleCategorySelect
 *
 *    Purpose:       Handle selection of category popup trigger
 *
 *    Parameters:    
 *
 *    Return Value:  none
 *
 *
 *************************************************************************/
static void DocListHandleCategorySelect(FormPtr frmP)
{
	Boolean categoryEdited;
	UInt16 category;
	
	category = gCurrentCategory;
	
	categoryEdited = DocListDBCategorySelect(frmP, DocListCategoryPopup,
	                  DocListCategoryList, true, &category,
	                  gCurrentCategoryName);
	   
	// Check if DocList needs to be updated               
	if (categoryEdited || (category != gCurrentCategory))
	{
		gCurrentCategory = category;
		
		// Force update of table
		FrmUpdateForm(DocListForm, frmUpdateEventReloadDBAndTable);
	}
}


/*************************************************************************
 *
 *    Function Name: DocListHandleControlSelect
 *
 *    Purpose:       Handle pushbutton select event to keep track of
 *                    which pushbutton is selected, and change table
 *                    highlight mode when in edit mode.
 *                    Also handle Category trigger selection
 *
 *    Parameters:    eventP: pointer to event
 *
 *    Return Value:  true if handled
 *
 *
 *************************************************************************/
static Boolean DocListHandleControlSelect(EventPtr eventP)
{
	FormPtr frmP;
	TablePtr tableP;
	Boolean handled;
	
	// Get pointer to current form and table
	frmP = FrmGetActiveForm();
	tableP = ControlGetPtrByID(frmP, DocListTable);
	
	handled = false;
	switch (eventP->data.ctlSelect.controlID)
	{
		case DocListCategoryPopup:
			DocListHandleCategorySelect(frmP);
			handled = true;
			break;
		case DocListRenameButton:
			// Edit indicator doesn't work, so turn it off while editing
			TblSetColumnEditIndicator(tableP, 0, false);
			gSelectedPushButton = eventP->data.ctlSelect.controlID;
			handled = true;
			break;
		case DocListOpenButton:
		case DocListDetailsButton:
		case DocListDeleteButton:
		case DocListBeamButton:
			// All other pushbuttons are handled same way
			TblSetColumnEditIndicator(tableP, 0, true);
			gSelectedPushButton = eventP->data.ctlSelect.controlID;
			handled = true;
			break;
	}
	return handled;
}

/*************************************************************************
 *
 *    Function Name: DocListHandleControlRepeat
 *
 *    Purpose:       Handle scroller buttons (repeatbutton type)
 *
 *    Parameters:    eventP: pointer to event
 *
 *    Return Value:  true if handled
 *
 *
 *************************************************************************/
static Boolean DocListHandleControlRepeat(EventPtr eventP)
{
	FormPtr frmP;
	TablePtr tableP;
	Boolean handled;
	
	// Get pointer to current form and table
	frmP = FrmGetActiveForm();
	tableP = ControlGetPtrByID(frmP, DocListTable);
	
	// Handled is left false so that we continually get events while
	//  button remains pressed
	handled = false;
	
	switch (eventP->data.ctlSelect.controlID)
	{
		case DocListScrollUpButton:
			DocListScroll(frmP, -1);
			break;
		case DocListScrollDownButton:
			DocListScroll(frmP, 1);
			break;
	}
	return handled;
}

/*************************************************************************
 *
 *    Function Name: DocListFormHandleEvent
 *
 *    Purpose:       Handle events for DocListForm
 *
 *    Parameters:    eventP: pointer to event to be processed
 *
 *    Return Value:  true only if handled
 *
 *
 *************************************************************************/
Boolean DocListFormHandleEvent (EventPtr eventP)
{
	Boolean handled = false;
	FormPtr frmP;
	TablePtr tableP;

	switch (eventP->eType)
	{
		case frmOpenEvent:
			// Initialize and draw form
			frmP = FrmGetActiveForm();
			DocListFormInit(frmP);
			FrmDrawForm(frmP);
			handled = true;
			break;
		
		case frmUpdateEvent:
			if ((frmUpdateEventReloadTable == eventP->data.frmUpdate.updateCode) ||
			    (frmUpdateEventReloadDBAndTable ==
			     eventP->data.frmUpdate.updateCode))
			{
				// Get pointers
				frmP = FrmGetActiveForm();
				tableP = ControlGetPtrByID(frmP, DocListTable);

				// Release focus in case currently editing
				TblReleaseFocus(tableP);

				// Refresh DocListDB and table
				if (frmUpdateEventReloadDBAndTable ==
				    eventP->data.frmUpdate.updateCode)
				{
					DocListDBPopulate();
				}
				DocListLoadTable(tableP);
				
				// Redraw table
				TblRedrawTable(tableP);
				handled = true;
			}
			else
			{
				handled = false;
			}
			break;
		
		case frmCloseEvent:
			DocListFormClose();
			// Handled = false because system needs to finish closing form
			handled = false;
			break;

	   case tblSelectEvent:
	   	// A document was selected so do appropriate action
	      handled = DocListHandleTableSelection(eventP);
	      break;
	
		case ctlSelectEvent:
			// Keep track of which push button is selected & handle category trigger
			handled = DocListHandleControlSelect(eventP);
			break;
	
		case ctlRepeatEvent:
			handled = DocListHandleControlRepeat(eventP);
			break;
	
	   case menuEvent:
			handled = MenuDocList(eventP);
			break;

	   case keyDownEvent:
	   	// Get pointer to form
			frmP = FrmGetActiveForm();
			// Scroll if hardware up/down key pressed
			switch (eventP->data.keyDown.chr)
			{
				case pageUpChr:
					DocListScroll(frmP, -1);
					handled = true;
					break;
				
				case pageDownChr:
					DocListScroll(frmP, 1);
					handled = true;
					break;
				
				default:
					break;
			}
		   break;
	
		default:
			break;
    }
    return handled;
}

