/*************************************************************************
 *
 *  File Name:   convert.c
 *
 *  Description: Functions for compressing and decompressing a Doc record
 *
 *  Special Thanks to Paul J. Lucas. Some of this code was adapted from
 *  txt2pdbdoc: Copyright (C) 1998 by Paul J. Lucas. Also a GPL program.
 *
 *  History:
 *     20 Mar 2000: Created; Jason Sherrill
 *     30 Mar 2000: Added comment headers; Jason Campbell
 *     15 Apr 2000: Created Compress; Jason Sherrill
 *     16 Apr 2000: Created MemFind; Jason Sherrill
 *     16 Apr 2000: Created PutByte; Jason Sherrill
 *     24 Apr 2000: Converted Compress & PutByte to use DmSet; Jason Sherrill
 *
 *************************************************************************
 * Copyright (C) 2000 Jason Campbell, Joshua Colvin, Jason Sherrill,
 *					Ben Tobin
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, In., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 *************************************************************************/

#include <PalmOS.h> 
#include "convert.h"


// Private function prototypes
static UInt8* MemFind(UInt8* scnPtr, Int16 scnLen, UInt8* strPtr, Int16 strLen);
static void PutByte(MemPtr b, UInt16* off, UInt8 c, Boolean* space);


/*************************************************************************
 *
 *    Function Name: Decompress
 *
 *    Purpose:       Compress the input buffer into the output buffer 
 *
 *    Parameters:    inBuf:     The input buffer
 *                   compLen:   Compressed length of the input buffer
 *                   outBuf:    The output buffer
 *                   decompLen: Size of the output buffer
 *
 *    Return Value:  compressed size in outBuf
 *
 *************************************************************************/

UInt16 Decompress(UInt8* inBuf, UInt16 compLen, UInt8* outBuf, UInt16 decompLen)
{
	UInt8  c;
	UInt16 c2;
	UInt16 m;
	UInt8  n;
	UInt8* copyFrom;
	UInt8* pastOutBuf;
	UInt8* pastInBuf;
	UInt8* startAddr;

	startAddr = outBuf;
	pastOutBuf = &outBuf[decompLen];
	pastInBuf = &inBuf[compLen];

	while (inBuf < pastInBuf && outBuf < pastOutBuf)
	{
		// Take a byte from the input buffer
		c = *(inBuf++);

		// Type A command, copy next c bytes directly
		if ( c >= 1 && c <= 8 )
			while ( c-- )
			{
				if (inBuf >= pastInBuf || outBuf >= pastOutBuf)
					break;
				*(outBuf++) = *(inBuf++);
			}

		// 0,9...7F represent themselves, copy that byte directly
		else if ( c <= 0x7F )
			*(outBuf++) = c;

		// Type B command, sliding window sequence
		else if ( c >= 0x80 && c <= 0xBF )
		{
			if (inBuf >= pastInBuf)
				break;
			c2 = (c << 8) | *(inBuf++);
			m = (c2 & 0x3FFF) >> 3;
			n = (c2 & 0x0007) + 3;
			copyFrom = outBuf - m;
			while (n--)
			{
				if (copyFrom >= pastOutBuf || outBuf >= pastOutBuf)
					break;
				*(outBuf++) = *(copyFrom++);
			}
		}
		
		// Type C command, copy a space + char formed by lower 7 bits
		else if (c >= 0xC0)
		{
			*(outBuf++) = ' ';
			if (outBuf >= pastOutBuf)
				break;
			*(outBuf++) = c & 0x7F;
		}

		else
			ErrDisplay("Error: Encountered strange byte in decompress");
	}
	
	return (outBuf - startAddr);	// Decompressed length
}


/*************************************************************************
 *
 *    Function Name: Compress
 *
 *    Purpose:       Compress the input buffer into the output DM record
 *
 *    Everywhere I had to insert a DmSet to write to Palm storage memory
 *    I commented out the non-system-specific line to do the same thing
 *    with pointers just below the DmSet. Same in putByte()
 *
 *    Parameters:    inBuf:      The input buffer
 *                   decompLen:  Decompressed length of the input buffer
 *                   outBuf:     The output buffer
 *                   maxCompLen: Maximum size of the output buffer
 *
 *    Return Value:  compressed size in outBuf
 *
 *************************************************************************/

// Something isn't working right, changed docview.c to not save anything
UInt16 Compress(MemPtr inBuf, UInt16 decompLen, MemPtr outBuf, UInt16 maxCompLen)
{
	UInt16 i, j;
	Boolean space = false;	// space flag
	UInt8* p;               // walking test hit; works up on successive matches  
	UInt8* p_prev;
	UInt8* head;            // current test string  
	UInt8* tail;            // 1 past the current test buffer
	UInt8* end;             // 1 past the end of the input buffer
	UInt16 dist;
	UInt16 compound;

	UInt8* outBufRd;
	UInt16 pos;             // position in the output buffer

	p = p_prev = head = inBuf;  // (UInt8*) cast?
	tail = head + 1;
	end = inBuf + decompLen;

	outBufRd = (UInt8*) outBuf;
	pos = 0;

	// loop, absorbing one more char from the input buffer on each pass  
	while ( head < end && pos < maxCompLen)
	{
		// establish where the scan can begin  
		if ( head - p_prev > (( 1 << 11 )-1) )
			p_prev = head - (( 1 << 11 )-1);

		// scan in the previous data for a match  
		p = MemFind( p_prev, tail - p_prev, head, tail - head );

		// on a mismatch or end of buffer, issued codes  
		if ( !p || p == head || tail - head > ( 1 << 3 ) + 2 || tail == end )
		{
			// issued the codes  
			// first, check for short runs  
			if ( tail - head < 4 )
				PutByte( outBuf, &pos, *head++, &space );
			else
			{
				dist = head - p_prev;
				compound = (dist << 3)	+ tail - head - 4;

				if ( dist >= ( 1 << 11 ) || tail - head - 4 > 7 )
					ErrDisplay("Error: dist overflow in compress()");

				// for longer runs, issue a run-code  
				// issue space char if required  
				if ( space )
					DmSet(outBuf, pos++, 1, ' ');
//					outBuf[pos++] = ' ', space = false;

				DmSet(outBuf, pos++, 1, 0x80 + ( compound >> 8 ));
//				outBuf[pos++] = 0x80 + ( compound >> 8 );
				DmSet(outBuf, pos++, 1, compound & 0xFF);
//				outBuf[pos++] = compound & 0xFF;
				head = tail - 1;// and start again  
			}
			p_prev = inBuf;	// start search again  
		} 
		else
			p_prev = p;		// got a match  

		// when we get to the end of the buffer, don't inc past the  
		// end; this forces the residue chars out one at a time  
		if ( tail != end )
			++tail;
	}

	if ( space )
		DmSet(outBuf, pos++, 1, ' ');
//		outBuf[pos++] = ' ';	// add left-over space  

	// final scan to merge consecutive high chars together  
	for ( i = j = 0; i < pos; ++i, ++j ) {
		DmSet(outBuf, j, 1, outBufRd[ i ]);
//		outBuf[ j ] = outBuf[ i ];

		// skip run-length codes  
		if ( outBufRd[ j ] >= 0x80 && outBufRd[ j ] < 0xC0 )
			DmSet(outBuf, ++j, 1, outBufRd[ ++i ]);
//			outBuf[ ++j ] = outBuf[ ++i ];

		// if we hit a high char marker, look ahead for another  
		else if ( outBufRd[ j ] == 1 )
		{
			DmSet(outBuf, j + 1, 1, outBufRd[ i + 1 ]);
//			outBuf[ j + 1 ] = outBuf[ i + 1 ];
			while ( i + 2 < pos && outBufRd[ i + 2 ] == 1 && outBufRd[ j ] < 8 )
			{
				DmSet(outBuf, j, 1, outBufRd[ j ] + 1);
//				outBuf[ j ]++;
				DmSet(outBuf, j + outBufRd[ j ], 1, outBufRd[ i + 3 ]);
//				outBuf[ j + outBuf[ j ] ] = outBuf[ i + 3 ];
				i += 2;
			}
			j += outBufRd[ j ];
			++i;
		}
	}
	return j;
}


/*************************************************************************
 *
 *    Function Name: MemFind
 *
 *    Purpose:       Scan in previous data for a match 
 *
 *    Parameters:    scnPtr:  address to start scan at
 *                   scnLen:  length to cover with scan
 *                   strPtr:  start address of string to scan for
 *                   strLen:  length of string to scan for
 *
 *    Return Value:  start of address where string was found
 *                   NULL if not found
 *
 *************************************************************************/

static UInt8* MemFind(UInt8* scnPtr, Int16 scnLen, UInt8* strPtr, Int16 strLen)
{
	Int16 i;

	for (i = scnLen - strLen + 1; i > 0; --i, ++scnPtr)
		if (*scnPtr == *strPtr && !MemCmp(scnPtr, strPtr, strLen))
			return scnPtr;

	return NULL;
}


/*************************************************************************
 *
 *    Function Name: PutByte
 *
 *    Purpose:       Put a byte into a buffer following 
 *
 *    Parameters:    b:      The output buffer
 *                   off:    The offset in the buffer to write to
 *                   c:      The byte to write
 *                   space:  Space flag for type C command
 *
 *    Return Value:  none
 *
 *************************************************************************/

static void PutByte(MemPtr b, UInt16* off, UInt8 c, Boolean* space)
{
	// If there is an outstanding space char: see if we can 
	// squeeze it in with an ASCII char
	if (*space)
	{
		if (c >= 0x40 && c <= 0x7F)
		{
			DmSet(b, *off, 1, c ^ 0x80);
//			b[*off] = c ^ 0x80;
			*off += 1;
		}
		else	// couldn't squeeze it in
		{
			DmSet(b, *off, 1, ' ');
//			b[*off] = ' ';	// write space char
			*off += 1;

			if (c < 0x80 && (c == 0 || c > 8))
			{
				DmSet(b, *off, 1, c);
//				b[*off] = c;
				*off += 1;
			}
			// chars in the range 1..8 or 0x80..0xFF need a one in front
			else
			{
				DmSet(b, *off, 1, 1);
//				b[*off] = 1;
				*off += 1;
				DmSet(b, *off, 1, c);
//				b[*off] = c;
				*off += 1;
			}
		}
		*space = false;	// clear the space flag
	}
	else
	{
		if (c == ' ')
			*space = true;	// set space flag, don't copy byte
		else
		{
			if (c < 0x80 && (c == 0 || c > 8))
			{
				DmSet(b, *off, 1, c);
//				b[*off] = c;
				*off += 1;
			}
			// chars in the range 1..8 or over 0x80 need a one in front
			else
			{
				DmSet(b, *off, 1, 1);
//				b[*off] = 1;
				*off += 1;
				DmSet(b, *off, 1, c);
//				b[*off] = c;
				*off += 1;
			}
		}
	}
}



