const { expect } = require('chai');
const sinon = require('sinon');
const fs = require('fs');
const { getInstalledSocialChannelsList, modifyXMLAndDownload } = require('../../../../features/social_integrations/migrations/_socialSetup.js');

describe('social channel functions', () => {
    let loggerMock, xmlFileName, siteConfigSocialChannel, siteInfo;

    beforeEach(() => {
        loggerMock = {
            info: sinon.spy(),
            debug: sinon.spy(),
        };

        xmlFileName = 'bm_extensions.xml';
        siteConfigSocialChannel = 'instagram';
        siteInfo = {
            data: {
                cartridges: 'int_instagram:int_google:int_tiktok',
            },
        };

        // Mock fs.readFile and fs.writeFile
        sinon.stub(fs, 'readFile').yields(null, '<menuaction id="social_channel_instagram" site="false"></menuaction>');
        sinon.stub(fs, 'writeFile').yields(null);
    });

    afterEach(() => {
        sinon.restore(); // Restore original methods
    });

    it('should update XML and write to file', async () => {
        await modifyXMLAndDownload({ logger: loggerMock }, xmlFileName, siteConfigSocialChannel, siteInfo);

        // Check logger calls
        expect(loggerMock.info.calledWith('Update bm_extensions.xml file to enable relevant menu actions.')).to.be.true;
        expect(loggerMock.info.calledWith('File bm_extensions.xml successfully updated!')).to.be.true;

        // Check that fs.readFile was called with the correct arguments
        expect(fs.readFile.calledWith(xmlFileName, 'utf8')).to.be.true;

        // Check that XML file was modified and written
        expect(fs.writeFile.calledWith(xmlFileName, sinon.match.any, 'utf8')).to.be.true;
    });

    it('should retrieve installed social channels list and update status map', () => {
        const socialChannelStatus = new Map([
            ['instagram', false],
            ['snapchat', false],
            ['google', false],
            ['tiktok', false],
        ]);

        getInstalledSocialChannelsList({ logger: loggerMock }, siteInfo, socialChannelStatus);

        // Check logger calls
        expect(loggerMock.info.calledWith('Retrieve already installed social channels in the site')).to.be.true;

        // Check socialChannelStatus map has been updated based on the cartridges in siteInfo
        expect(socialChannelStatus.get('instagram')).to.be.true;
        expect(socialChannelStatus.get('tiktok')).to.be.true;
        expect(socialChannelStatus.get('snapchat')).to.be.false; // Not in cartridges, so should remain false
        expect(socialChannelStatus.get('google')).to.be.true;
    });
});