import { test, expect, request } from '@playwright/test';

function delay(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
  }

async function jobExecutionRequest(process, jobID){
    const url = 'https://account.demandware.com/dw/oauth2/access_token';
    const clientId = process.env.INSTAGRAM_OCAPI_CLIENT_ID;
    const clientSecret = process.env.INSTRAGRAM_OCAPI_CLIENT_SECRET;
    const grantType = 'client_credentials';
    const authHeader = `Basic ${Buffer.from(`${clientId}:${clientSecret}`).toString('base64')}`;
    const data = new URLSearchParams();
    data.append('grant_type', grantType);

    const response = await fetch(url, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded',
        'Authorization': authHeader,
      },
      body: data,
    });

    const dataJson = await response.json();
    const accessToken = dataJson.access_token;
    
    if(!accessToken){
      throw new Error(dataJson.error + ": " + dataJson.error_description);
    }
    console.log('Successful authentication...');

    const responseJobExecution = await fetch(process.env.BM_URL + '/s/Sites-Site/dw/data/v21_3/jobs/'+jobID+'/executions', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': 'Bearer ' + accessToken,
      },
      body: null,
    });

    const responseJobExecutionParsed = await responseJobExecution.json();

    if(responseJobExecutionParsed.fault){
      throw new Error(responseJobExecutionParsed.fault.type + ": " + responseJobExecutionParsed.fault.message);
    }

    const getStatusUrl = process.env.BM_URL + '/s/Sites-Site/dw/data/v21_3/jobs/'+jobID+'/executions/' + responseJobExecutionParsed.id;

    let status = 'RUNNING';

    while(status !=='OK'){
      const responseJobExecutionStatus = await fetch(getStatusUrl, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': 'Bearer ' + accessToken,
        },
        body: null,
      });

      const responseJobExecutionStatusJson = await responseJobExecutionStatus.json();
      status = responseJobExecutionStatusJson.status;

      if(status === 'RUNNING'){
        await delay(1000);
      }
    }

    return status;
}

test.describe('Create order and get oci inventory.', () => {
    test.beforeAll(async ({ request }) => {
        //TSOB Token
        const accessToken = await request.post(`https://${process.env.BM_SHORT_CODE}.api.commercecloud.salesforce.com/shopper/auth/v1/organizations/${process.env.BM_ORGANIZATION_ID}/oauth2/trusted-system/token`, {
            headers: {
                "Content-Type": "application/x-www-form-urlencoded",
                'Authorization': `Basic ${Buffer.from(process.env.SLAS_CLIENTID + ':' + process.env.SLAS_CLIENTSECRET).toString("base64")}`
            },
            form: {
                "grant_type": "client_credentials",
                "hint": "ts_ext_on_behalf_of",
                "login_id": process.env.INSTAGRAM_EMAIL,
                "idp_origin": process.env.SOCIAL_CHANNEL,
                "channel_id": process.env.BM_SITE
            }
        });

        expect(accessToken.ok()).toBeTruthy();

        const response = await accessToken.json();

        process.env.ACCESS_TOKEN = response.access_token;
    });

    test('Scenario 1 - Get a TSOB token and create order', async ({ request }) => {
         await test.step('Can create an order on-the-fly in the Commerce Cloud platform', async () => {
            let response = await request.post(`https://${process.env.BM_SHORT_CODE}.api.commercecloud.salesforce.com/checkout/orders/v1/organizations/${process.env.BM_ORGANIZATION_ID}/orders?siteId=${process.env.BM_SITE}`, {
                headers: {
                    "Content-Type": "application/json",
                    "Authorization": `Bearer ${process.env.ACCESS_TOKEN}`
                },
                data: JSON.stringify({
                    "billingAddress": {
                    "address1": "43 Main Rd.",
                    "city": "Burlington",
                    "firstName": "Vitoria Carolina",
                    "lastName": "Santos"
                  },
                  "channelType": "instagramcommerce",
                  "currency": "USD",
                  "orderTotal": 355.39,
                  "taxTotal": 44.78,
                 "c_buyerNote": "note from buyer",
                 "c_orderNote": "note for the order",
                 "c_orderTag": "Tag for the order",
                  "paymentInstruments": [
                    {
                      "paymentMethodId": "PAYPAL",
                      "paymentTransaction": {
                        "amount": 303.95,
                        "transactionId": "abc13384ajsgdk1"
                      }
                    }
                  ],
                  "productItems": [
                    {
                      "basePrice": 299.00,
                      "grossPrice": 299.00,
                      "netPrice": 299.00,
                      "productId": "609717696966M",
                      "productName": "5 Pocket Cuffed Capri (Plus)",
                      "quantity": 2,
                      "shipmentId": "shipment1",
                      "tax": 12.39
                    },
                    {
                      "basePrice": 24.00,
                      "grossPrice": 24.00,
                      "netPrice": 24.00,
                      "productId": "701642923466M",
                      "productName": "3/4 Sleeve V-Neck Top",
                      "quantity": 1,
                      "shipmentId": "shipment1",
                      "tax": 10.00
                    }
                  ],
                  "shipments": [
                    {
                      "shipmentId": "shipment1",
                      "shippingAddress": {
                        "address1": "43 Main Rd.",
                        "city": "Burlington",
                        "firstName": "Vitoria Carolina",
                        "lastName": "Santos"
                      },
                      "shippingMethod": "001",
                      "shippingTotal": 10.00,
                      "taxTotal": 22.39
                    }
                  ]
                 })
            });

            expect(response.ok()).toBeTruthy();
            //Run OCI changed product job
            const resultOCI = await jobExecutionRequest(process,"OCI-Download-Delta-Inventory");
            expect(resultOCI).toEqual("OK");
        });
    });
});
