public class ReturnOrderLineItemTriggerHandler {
    private static final String ORDER_ACTION = 'update';
    private static final Map<String, Integer> returnStatusMap = new Map<String, Integer>{
        'CANCELLED' => 6,
        'LINE_CANCELLED' => 3,
        'PARTIAL_CANCEL' => 7,
        'RETURNED' => 4,
        'PARTIAL_RETURNED' => 5
    };
    private static final String PRODUCT = 'Order Product';
    private static final String DELIVERY = 'Delivery Charge';
    private static final String RETURN_FEE = 'Fee';
    private static final String ORDER_CHANGE_RETURN_TYPE = 'Return';
    private static final Set<String> LINE_ITEM_TYPE = new Set<String>{DELIVERY, PRODUCT};


    public static void handleReturnRequest (List<ReturnOrderLineItem> triggerList){
        Map<String, List<String>> orderNumberPayloadReturnMap = sendReturnUpdate(triggerList);
        Map<String, List<String>> orderNumberPayloadFeeMap = sendReturnFeeUpdate(triggerList);

        Map<String, List<String>> orderNumbersToPayloads = mergeMaps(orderNumberPayloadReturnMap, orderNumberPayloadFeeMap);

        if (orderNumbersToPayloads.size() > 0) {
            HttpResponse res = new HttpResponse();

            for(String orderNumber : orderNumbersToPayloads.keySet()){
                String payload = mergePayloads(orderNumbersToPayloads.get(orderNumber));
                sendOrderUpdate(orderNumber, payload);
            }
        }

    }

    public static Map<String, List<String>> mergeMaps(Map<String, List<String>> orderNumberPayloadReturnMap, Map<String, List<String>> orderNumberPayloadFeeMap){
        for (String orderNumber : orderNumberPayloadFeeMap.keySet()) {
            List<String> currentOrderNumberPayloadReturnMap = orderNumberPayloadReturnMap.get(orderNumber);
            List<String> currentOrderNumberPayloadFeeMap = orderNumberPayloadFeeMap.get(orderNumber);

            if(currentOrderNumberPayloadReturnMap != null){
                currentOrderNumberPayloadReturnMap.addAll(currentOrderNumberPayloadFeeMap);
                orderNumberPayloadReturnMap.put(orderNumber, currentOrderNumberPayloadReturnMap);
            } else {
                List<String> orderPayloads = new List<String>();
                orderPayloads.addAll(currentOrderNumberPayloadFeeMap);
                orderNumberPayloadReturnMap.put(orderNumber, orderPayloads);
            }
        }
        return orderNumberPayloadReturnMap;
    }

    public static String mergePayloads(List<String> payloads){
        if(payloads.size() > 1){
            List<Object> productItems = new List<Object>();
            for (String payload : payloads) {
                Map<String, Object> deserializedPayload = (Map<String, Object>) JSON.deserializeUntyped(payload);
                productItems.addAll((List<Object>) deserializedPayload.get('product_items'));
            }

            Map<String, Object> deserializeUntypedResponse = (Map<String, Object>) JSON.deserializeUntyped(payloads[0]);
            deserializeUntypedResponse.put('product_items', productItems);

            return JSON.serialize(deserializeUntypedResponse);
        }
        return payloads[0];
    }

    public static Map<String, List<String>> sendReturnUpdate(List<ReturnOrderLineItem> triggerList){
        Map<Id, Map<ReturnOrderLineItem, String>> lineItemReturnMap = new Map<Id, Map<ReturnOrderLineItem, String>>();
        Map<Id, String> returnAndSummaryMap = new Map<Id, String>();
        Map<Id, Integer> mapOrderSummaryIdToOrderProductLineCount = new Map<Id, Integer>();
        Map<Id, Id> mapReturnAndSummary = new Map<Id, Id>();
        Map<Id, Integer> mapReturnIdToReturnProductShipCount = new Map<Id, Integer>();
        Set<Id> returnOrderIdsWithReturnFee = new Set<Id>();
        Set<Id> returnOrderIds = new Set<Id>();


        for (ReturnOrderLineItem line : triggerList) {
            returnOrderIds.add(line.returnOrderId);
        }

        for (ReturnOrder ret : [SELECT ID, OrderSummaryId FROM ReturnOrder WHERE ID IN : returnOrderIds]) {
            mapReturnAndSummary.put(ret.id, ret.OrderSummaryId);
        }

        Map<Id, OrderItemSummary> orderItemSummaryMap = new Map<Id, OrderItemSummary>(
                                                                [SELECT Id, Type, QuantityFulfilled, QuantityReturned, OrderSummaryId, ProductCode
                                                                FROM OrderItemSummary
                                                                WHERE OrderSummaryId IN :mapReturnAndSummary.values()
                                                                AND Type IN :LINE_ITEM_TYPE]
                                                            );


        for (OrderSummary orderSum : [SELECT ID, OrderNumber FROM OrderSummary WHERE ID IN : mapReturnAndSummary.values()]) {
            returnAndSummaryMap.put(orderSum.id, orderSum.OrderNumber);
        }

        for (OrderItemSummary item : orderItemSummaryMap.values()) {
            Id orderSummaryId = item.OrderSummaryId;
            Integer lineCount = mapOrderSummaryIdToOrderProductLineCount.containsKey(orderSummaryId)
                                        ? mapOrderSummaryIdToOrderProductLineCount.get(orderSummaryId) + 1:1;

            mapOrderSummaryIdToOrderProductLineCount.put(orderSummaryId,lineCount);
        }

        for (ReturnOrderLineItem line : triggerList) {

            if(line.QuantityRejected == 0 && line.QuantityReceived == 0 ) {
                continue;
            }

            String status;
            OrderItemSummary summary = orderItemSummaryMap.get(line.orderItemSummaryId);

            Decimal quantityFulfilled = (summary != null) ? summary.QuantityFulfilled : 0;

            if (line.QuantityRejected > 0) {
                status = (line.QuantityRejected == quantityFulfilled) ? 'LINE_CANCELLED' : 'PARTIAL_CANCEL';
            } else if (line.QuantityReturned > 0) {
                status = (line.QuantityReturned == quantityFulfilled) ? 'RETURNED' : 'PARTIAL_RETURNED';
            }
            Map<ReturnOrderLineItem, String> innerMap = lineItemReturnMap.containsKey(line.ReturnOrderId)
                                                        ? lineItemReturnMap.get(line.ReturnOrderId) : new Map<ReturnOrderLineItem, String>();
            innerMap.put(line, status);
            lineItemReturnMap.put(line.ReturnOrderId, innerMap);
        }

        for (ReturnOrderLineItem returnOrderLineItem : [SELECT ID, ReturnOrderId FROM ReturnOrderLineItem
                                                        WHERE ReturnOrderId IN : mapReturnAndSummary.keySet()
                                                        AND Type IN :LINE_ITEM_TYPE]) {
                Integer returnItemIds = mapReturnIdToReturnProductShipCount.get(returnOrderLineItem.ReturnOrderId) != null ?
                mapReturnIdToReturnProductShipCount.get(returnOrderLineItem.ReturnOrderId) : 0;
                mapReturnIdToReturnProductShipCount.put(returnOrderLineItem.ReturnOrderId, returnItemIds+1);
        }

        for (ReturnOrderLineItem returnOrderLineItem : [
            SELECT ReturnOrderId
            FROM ReturnOrderLineItem
            WHERE ReturnOrderId IN :mapReturnAndSummary.keySet()
            AND Type = :RETURN_FEE
        ]) {
            returnOrderIdsWithReturnFee.add(returnOrderLineItem.ReturnOrderId);
        }

        for (Id returnOrderId : lineItemReturnMap.keySet()) {
            Map<ReturnOrderLineItem, String> innerMap = lineItemReturnMap.get(returnOrderId);

            for (ReturnOrderLineItem lineItem : innerMap.keySet()) {
                String status = innerMap.get(lineItem);
            }
        }

        Map<String, Map<String, String>> mapReturnToPayloadAndOrder = new Map<String, Map<String, String>>();
        Map<String, List<String>> mapOrderToPayloads = new Map<String, List<String>>();
        for (Id returnId : lineItemReturnMap.keySet()) {
            String payload='';
            Id currOrderSummaryId = mapReturnAndSummary.get(returnId);
            JSONGenerator gen = JSON.createGenerator(true);
                gen.writeStartObject();
                gen.writeStringField('c_orderAction', ORDER_ACTION);
                if (lineItemReturnMap.get(returnId).values().contains('PARTIAL_CANCEL')) {
                    gen.writeNumberField('c_externalReturnStatus',returnStatusMap.get('PARTIAL_CANCEL') );
                }else if(lineItemReturnMap.get(returnId).values().contains('PARTIAL_RETURNED')){
                    gen.writeNumberField('c_externalReturnStatus',returnStatusMap.get('PARTIAL_RETURNED') );
                }else if(lineItemReturnMap.get(returnId).values().contains('LINE_CANCELLED')){
                    if(mapReturnIdToReturnProductShipCount.get(returnId) == mapOrderSummaryIdToOrderProductLineCount.get(currOrderSummaryId)){
                        gen.writeNumberField('c_externalReturnStatus',returnStatusMap.get('CANCELLED') );
                    }
                    else {
                        gen.writeNumberField('c_externalReturnStatus',returnStatusMap.get('PARTIAL_CANCEL') );
                    }
                }else if(lineItemReturnMap.get(returnId).values().contains('RETURNED')){
                    if(mapReturnIdToReturnProductShipCount.get(returnId) == mapOrderSummaryIdToOrderProductLineCount.get(currOrderSummaryId)
                        && !returnOrderIdsWithReturnFee.contains(returnId)){
                        gen.writeNumberField('c_externalReturnStatus',returnStatusMap.get('RETURNED') );
                    }
                    else {
                        gen.writeNumberField('c_externalReturnStatus',returnStatusMap.get('PARTIAL_RETURNED') );
                    }
                }

                gen.writeFieldName('product_items');
                gen.writeStartArray();
                for(ReturnOrderLineItem item : lineItemReturnMap.get(returnId).keySet()){
                    if (orderItemSummaryMap.containskey(item.OrderItemSummaryId)) {
                        if(orderItemSummaryMap.get(item.OrderItemSummaryId).Type == PRODUCT){
                            gen.writeStartObject();
                            gen.writeStringField('product_id',orderItemSummaryMap.get(item.OrderItemSummaryId).ProductCode);

                            gen.writeNumberField('c_quantityReceived', item.QuantityReceived);
                            gen.writeNumberField('c_quantityReturned', item.QuantityReturned);
                            gen.writeNumberField('c_quantityRejected', item.QuantityRejected);

                            if (returnStatusMap.containsKey(lineItemReturnMap.get(returnId).get(item))) {
                                gen.writeNumberField('c_externalLineItemReturnStatus',returnStatusMap.get(lineItemReturnMap.get(returnId).get(item)));
                            }
                            gen.writeEndObject();
                        }
                    }
                }
                gen.writeEndArray();
                gen.writeEndObject();

            payload = gen.getAsString();

            String orderSummaryId = mapReturnAndSummary.get(returnId);
            String orderNumber = returnAndSummaryMap.get(orderSummaryId);

            updateOrdItemSumChanges(orderSummaryId, orderItemSummaryMap);
            List<String> arrayOfPayloads = mapOrderToPayloads.get(orderNumber);

            if(arrayOfPayloads != null) {
                arrayOfPayloads.add(payload);
            } else {
                arrayOfPayloads = new List<String>();
                arrayOfPayloads.add(payload);
            }

            mapOrderToPayloads.put(orderNumber, arrayOfPayloads);
        }

        return mapOrderToPayloads;
    }

    public static Map<String, List<String>> sendReturnFeeUpdate(List<ReturnOrderLineItem> triggerList){
        Map<Id, Map<ReturnOrderLineItem, Id>> lineItemReturnMap = new Map<Id, Map<ReturnOrderLineItem, Id>>();
        Map<Id, String> OrderSummaryMap = new Map<Id, String>();
        Map<Id, Id> mapReturnOrderSummary = new Map<Id, Id>();
        Map<Id, OrderItemSummary> orderItemFeeSum = new Map<Id, OrderItemSummary>();
        Set<Id> OrderItemSummaryIds = new Set<Id>();
        Map<ReturnOrderLineItem, Id> innerMap = new Map<ReturnOrderLineItem, Id>();

        for (ReturnOrderLineItem line : triggerList) {
            if(line.OrderItemSummaryId != null && line.Type == RETURN_FEE){
                Id returnOrderIdKey = line.ReturnOrderId;
                innerMap.put(line, line.Id);
                lineItemReturnMap.put(returnOrderIdKey, innerMap);
            }
        }

        Map<String, List<String>> mapOrderToPayloads = new Map<String, List<String>>();
        if(!lineItemReturnMap.isEmpty()){
            for (ReturnOrder ret : [SELECT ID, OrderSummaryId, Status FROM ReturnOrder WHERE ID IN : lineItemReturnMap.keySet()]) {
                if(ret.Status == 'Closed'){
                    mapReturnOrderSummary.put(ret.id, ret.OrderSummaryId);
                }
            }

            for (OrderSummary orderSum : [SELECT ID, OrderNumber FROM OrderSummary WHERE ID IN : mapReturnOrderSummary.values()]) {
                OrderSummaryMap.put(orderSum.id, orderSum.OrderNumber);
            }

            for (Id returnOrderId : lineItemReturnMap.keySet()) {
                for (ReturnOrderLineItem returnLine : lineItemReturnMap.get(returnOrderId).keySet()) {
                    if(returnLine.OrderItemSummaryId != null){
                        OrderItemSummaryIds.add(returnLine.OrderItemSummaryId);
                    }
                }
            }

            for (OrderItemSummary orderItem : [SELECT ID, ProductCode, Type, Status FROM OrderItemSummary WHERE ID IN: OrderItemSummaryIds]) {
                orderItemFeeSum.put(orderItem.id, orderItem);
            }

            String orderNumber = '';
            String payload='';
            for (Id returnId : lineItemReturnMap.keySet()) {
                Id orderItemSummaryId = mapReturnOrderSummary.get(returnId);
                if (orderItemSummaryId != null) {
                    payload='';
                    JSONGenerator gen = JSON.createGenerator(true);
                    gen.writeStartObject();
                    gen.writeStringField('c_orderAction', ORDER_ACTION);
                    gen.writeFieldName('product_items');
                    gen.writeStartArray();
                    Map<ReturnOrderLineItem, Id> lineId = lineItemReturnMap.get(returnId);
                    Map<String, Decimal> feeSum = new Map<String, Decimal>();
                    String product2Id = '';
                    for(ReturnOrderLineItem line : lineId.keySet()){
                        if (orderItemFeeSum.containskey(line.OrderItemSummaryId)) {
                            if(line.Type == RETURN_FEE){
                                product2Id = orderItemFeeSum.get(line.OrderItemSummaryId).ProductCode;
                                Decimal feeFixedNegative = line.TotalAmount;
                                if(line.TotalAmount < 0){
                                    feeFixedNegative = line.TotalAmount * -1;
                                }
                                if (!feeSum.containsKey(product2Id)) {
                                    feeSum.put(product2Id, feeFixedNegative);
                                }else{
                                    Decimal fee = feeSum.get(product2Id);
                                    feeSum.put(product2Id, fee + feeFixedNegative);
                                }
                            }
                        }
                    }
                    gen.writeStartObject();
                    gen.writeStringField('product_id', product2Id);
                    Decimal fee = 0;
                    for(String line : feeSum.keySet()){
                        fee = fee + feeSum.get(line);
                    }
                    gen.writeNumberField('c_restockingFee', fee);
                    gen.writeEndObject();
                    gen.writeEndArray();
                    gen.writeEndObject();

                    payload = gen.getAsString();

                    orderNumber = OrderSummaryMap.get(mapReturnOrderSummary.get(returnId));
                    String orderNumberId = mapReturnOrderSummary.get(returnId);

                    List<String> arrayOfPayloads = mapOrderToPayloads.get(orderNumber);

                    if(arrayOfPayloads != null) {
                        arrayOfPayloads.add(payload);
                    } else {
                        arrayOfPayloads = new List<String>();
                        arrayOfPayloads.add(payload);
                    }

                    mapOrderToPayloads.put(orderNumber, arrayOfPayloads);
                }
            }
        }

        return mapOrderToPayloads;
    }

    public static void updateOrdItemSumChanges(String orderSummaryId,Map<Id, OrderItemSummary> orderItemSummaryMap){
        List<OrderItemSummaryChange> orderItemSummaryChanges = new List<OrderItemSummaryChange>([Select Id, IsDeleted, OrderItemSummaryChangeNumber, CreatedDate, CreatedById,
        LastModifiedDate, LastModifiedById, SystemModstamp, OrderItemSummaryId, ChangeOrderItemId, OrderSummaryId,
        Reason, ChangeType, IsPreFulfillment, ReasonText, IsChangeSynced__c
        from OrderItemSummaryChange
        where OrderSummaryId =: orderSummaryId and
        ChangeType =: ORDER_CHANGE_RETURN_TYPE and
        IsChangeSynced__c =: false]);
        List<OrderItemSummaryChange> ordItemSumChangesListToDelete = new List<OrderItemSummaryChange>();
        List<OrderItemSummaryChange> ordItemSumChangesListToAdd = new List<OrderItemSummaryChange>();

        for(OrderItemSummaryChange ordItemSumChange : orderItemSummaryChanges){
            if(ordItemSumChange.ChangeType == ORDER_CHANGE_RETURN_TYPE && orderItemSummaryMap.containsKey(ordItemSumChange.OrderItemSummaryId)
                && orderItemSummaryMap.get(ordItemSumChange.OrderItemSummaryId).Type == PRODUCT) {
                ordItemSumChangesListToDelete.add(ordItemSumChange);
            }
        }
        delete ordItemSumChangesListToDelete;

        for(OrderItemSummaryChange ordItemSumChange : ordItemSumChangesListToDelete){
            ordItemSumChange.Id = null;
            ordItemSumChange.IsChangeSynced__c = true;
            ordItemSumChangesListToAdd.add(ordItemSumChange);
        }
        insert ordItemSumChangesListToAdd;
    }

    @future(callout = true)
    public static void sendOrderUpdate(String orderNumber, String payload){
        OrderStatusExportAuthHelper authHelper = new OrderStatusExportAuthHelper();
        Order_Status_Update_Params__mdt params = authHelper.getOAuthConnInfo();
        String token = authHelper.getOAuthToken(params);

        authHelper.sendOrderUpdate(params, token, orderNumber, payload);
    }
}
