'use strict';

const LineItemCtnr = require('dw/order/LineItemCtnr');
const Logger = require('dw/system/Logger').getLogger('social-checkout', 'helpers');

/**
 * Returns LineItemCtnr.channelType constant from the social channel
 * @param {string|number} socialChannel - social channel as string (TikTok) or integer (14)
 * @returns {number|null} channel type number or null
 */
const getChannelType = function (socialChannel) {
    var channelType = null;
    if (!socialChannel) return channelType;
    try {
        channelType = parseInt(socialChannel, 10);
        // script API 18.10 doesn't have Number.isNaN
        // eslint-disable-next-line no-restricted-globals
        if ((Number.isNaN === undefined && isNaN(channelType)) || (Number.isNaN !== undefined && Number.isNaN(channelType))) {
            channelType = LineItemCtnr['CHANNEL_TYPE_' + String(socialChannel.toUpperCase())];
        }
    } catch (e) {
        channelType = null;
        Logger.error(e.toString() + ' in ' + e.fileName + ':' + e.lineNumber);
    }
    return channelType;
};

/**
 * Returns basketResponse.productItems additional infos about applied discount quantity (BXGY promotions)
 * @param {dw.order.Basket} basket - current basket
 * @param {Object} basketResponse  - basket response
 * @returns {Object} - productItems object
 */
const setPriceAdjustmentsAppliedDiscountQuantity = function (basket, basketResponse) {
    /* eslint-disable no-loop-func */
    var productItems = basketResponse.productItems;

    try {
        for (var i = 0; i < productItems.length; i++) {
            var productItem = productItems[i];
            var priceAdjustments = productItem.priceAdjustments;
            if (priceAdjustments && priceAdjustments.length > 0) {
                for (var j = 0; j < priceAdjustments.length; j++) {
                    var priceAdjustmentId = priceAdjustments[j].priceAdjustmentId;
                    var promotionID = priceAdjustments[j].promotionId;

                    var productLineItems = basket.allProductLineItems.toArray().filter(function (pli) {
                        return productItem.itemId === pli.UUID;
                    }, []);
                    var productLineItem = !empty(productLineItems) ? productLineItems[0] : null;
                    var basketPriceAdjustments = !empty(productLineItem) ? productLineItem.priceAdjustments.toArray().filter(function (pa) {
                        return pa && pa.UUID === priceAdjustmentId && pa.promotionID === promotionID;
                    }, []) : [];

                    var priceAdjustmentByPromotionID = !empty(basketPriceAdjustments) ? basketPriceAdjustments[0] : null;
                    var quantity = priceAdjustmentByPromotionID && priceAdjustmentByPromotionID.appliedDiscount && priceAdjustmentByPromotionID.appliedDiscount.quantity ? priceAdjustmentByPromotionID.appliedDiscount.quantity : null;
                    if (quantity != null) {
                        priceAdjustments[j].c_appliedDiscountQuantity = quantity;
                        if (!empty(productItem.quantity) && Number(productItem.quantity) !== quantity) {
                            priceAdjustments[j].c_appliedDiscountType = 'BXGY';
                        }
                    }
                }
            }
        }
    } catch (error) {
        return basketResponse.productItems;
    }

    return productItems;
};

module.exports = {
    getChannelType: getChannelType,
    setPriceAdjustmentsAppliedDiscountQuantity: setPriceAdjustmentsAppliedDiscountQuantity
};
