'use strict';

var TaxMgr = require('dw/order/TaxMgr');
var Status = require('dw/system/Status');
var collections = require('*/cartridge/scripts/util/collections');

/**
 * If the channel type of the basket is equal to 12 (instagram commerce), return false to prevent tax updates and avoid mismatching.
 * @param {dw.order.Basket} basket - basket
 * @returns {boolean} - if is allowed to update taxes
 */
function allowTaxesUpdates(basket) {
    if (empty(basket.channelType) || basket.channelType.value !== 12) {
        return true;
    }

    return false;
}

/**
 *
 * @function calculateTax <p>
 *
 * Determines tax rates for all line items of the basket. Uses the shipping addresses
 * associated with the basket shipments to determine the appropriate tax jurisdiction.
 * Uses the tax class assigned to products and shipping methods to lookup tax rates. <p>
 *
 * Sets the tax-related fields of the line items. <p>
 *
 * Handles gift certificates, which aren't taxable. <p>
 *
 * Note that the function implements a fallback to the default tax jurisdiction
 * if no other jurisdiction matches the specified shipping location/shipping address.<p>
 *
 * Note that the function implements a fallback to the default tax class if a
 * product or a shipping method does explicitly define a tax class.
 *
 * NEW: Not updating taxes for Instagram baskets
 *
 * @param {dw.order.Basket} basket The basket containing the elements for which taxes need to be calculated
 */

exports.calculateTax = function (basket) {
    var basketCalculationHelpers = require('*/cartridge/scripts/helpers/basketCalculationHelpers');

    var taxes = basketCalculationHelpers.calculateTaxes(basket);

    // convert taxes into hashmap for performance.
    var taxesMap = {};

    taxes.taxes.forEach(function (item) {
        taxesMap[item.uuid] = { value: item.value, amount: item.amount };
    });

    var lineItems = basket.getAllLineItems();

    var totalShippingGrossPrice = 0; // eslint-disable-line
    var totalShippingNetPrice = 0; // eslint-disable-line

    // update taxes for all line items
    collections.forEach(lineItems, function (lineItem) {
        var tax = taxesMap[lineItem.UUID];

        if (tax) {
            if (tax.amount) {
                lineItem.updateTaxAmount(tax.value);

                if (lineItem instanceof dw.order.ShippingLineItem) { // eslint-disable-line
                    totalShippingGrossPrice += lineItem.getAdjustedGrossPrice();
                    totalShippingNetPrice += lineItem.getAdjustedNetPrice();
                }
            } else {
                lineItem.updateTax(tax.value);
            }
        } else {
            /* eslint-disable */
            if (lineItem.taxClassID === TaxMgr.customRateTaxClassID) {
                // do not touch tax rate for fix rate items
                lineItem.updateTax(lineItem.taxRate);
            } else {
                // otherwise reset taxes to null
                lineItem.updateTax(null);
            }
            /* eslint-enable */
        }
    });

    // besides shipment line items, we need to calculate tax for possible order-level price adjustments
    // this includes order-level shipping price adjustments
    if (!basket.getPriceAdjustments().empty || !basket.getShippingPriceAdjustments().empty) {
        if (collections.first(basket.getPriceAdjustments(), function (priceAdjustment) {
            return taxesMap[priceAdjustment.UUID] === null;
        }) || collections.first(basket.getShippingPriceAdjustments(), function (shippingPriceAdjustment) {
            return taxesMap[shippingPriceAdjustment.UUID] === null;
        })) {
            // Not updating taxes for Instagram baskets
            if (allowTaxesUpdates(basket)) {
                basket.updateOrderLevelPriceAdjustmentTax();
            }
        }
    }

    // if hook returned custom properties attach them to the order model
    if (taxes.custom) {
        Object.keys(taxes.custom).forEach(function (key) {
            basket.custom[key] = taxes.custom[key];
        });
    }

    return new Status(Status.OK);
};
