'use strict';

var HookMgr = require('dw/system/HookMgr');
var Logger = require('dw/system/Logger');

var orderHelpers = require('~/cartridge/scripts/helpers/orderHelpers');
var som = require('~/cartridge/scripts/som');
var somPlatformCancellationHelpers = require('~/cartridge/scripts/helpers/somPlatformCancellationHelpers');

const DEFAULT_CANCEL_REASON = 'Unknown';

/**
 *   check if there is enough qty to cancel
 *   @param {Object} omsOrder - som order object
 *   @param {Object} sfccOrder - sfcc order object
 *   @param {Object} cancelItems - items to cancel
 *   @return {boolean} - return true if there was a status change
 */
function getCancelOrderItems(omsOrder, sfccOrder, cancelItems) {
    var items2Cancel = cancelItems.keySet().toArray();
    var omsLIs = omsOrder.orderItems.orderItems;
    var cancelJSON = null;
    var changeItemArr = [];
    omsLIs.forEach(function (omsLi) {
        items2Cancel.forEach(function (productID) {
            if (omsLi.sfccProductId === productID) {
                var itemCancel = cancelItems.get(productID);
                // check if qty available for cancellation
                if (omsLi.quantityAvailableToCancel) {
                    // add item to cancelJSON
                    var cancelItemObj = {};
                    cancelItemObj.id = omsLi.orderItemSummaryId;
                    cancelItemObj.quantity = itemCancel.get('quantity') > omsLi.quantityAvailableToCancel ? omsLi.quantityAvailableToCancel : itemCancel.get('quantity');
                    cancelItemObj.reason = itemCancel.get('reason');
                    changeItemArr.push(cancelItemObj);
                } else {
                    Logger.error('Not available qty to cancel for the productID ' + productID);
                }
            }
        });
    });
    if (changeItemArr.length > 0) {
        cancelJSON = {};
        cancelJSON.summaryId = omsOrder.orderSummaryId;
        cancelJSON.lineItems = changeItemArr;
        cancelJSON.orderId = sfccOrder.orderNo;
        cancelJSON.currencyCode = sfccOrder.getCurrencyCode();
    }
    return cancelJSON;
}

/**
*   cancel the given items by calling SOM API
*   @param {Object} order - sfcc order object
*   @param {Object} orderInput -  JSON of which items needs to be cancelled
*   @return {boolean} - return true if no API exceptions
*/
exports.processCancel = function (order, orderInput) {
    var HashMap = require('dw/util/HashMap');
    var orderID = order.getOrderNo();
    var somOrder = orderHelpers.getOrderSummary([orderID]);
    var cancelItems = orderInput.c_cancelInfo.get('cancelItems');
    var sfccOrders;
    // make sure the OMS order is not shipped
    if (somOrder != null && somOrder.length > 0 && somOrder[0].orderedStatusGroupItems != null && somOrder[0].orderedStatusGroupItems.length > 0) {
        if (order.custom.cancelInfo != null && order.custom.cancelInfo.indexOf('cancelItems') > 0) {
            var cancelData = getCancelOrderItems(somOrder[0].orderedStatusGroupItems[0], order, cancelItems);
            if (cancelData != null) {
                var somRes = som.cancelOrderItems(cancelData);
                if (somRes.ok && somRes.object.responseObj[0].isSuccess) {
                    // update SFCC order status
                    var orderNumbers = [];
                    sfccOrders = new HashMap();
                    orderNumbers.push(orderID);
                    sfccOrders.put(orderID, order);
                    HookMgr.callHook('app.order.update.processStatusUpdate', 'processStatusUpdate', JSON.stringify(orderNumbers), sfccOrders);
                    return true;
                }
                Logger.error('Error cancelling order items in SOM');
            }
        } else {
            Logger.error('No items to cancel in the request');
            return false;
        }
    }

    var items2Cancel = !empty(cancelItems) ? cancelItems.keySet().toArray() : [];
    var itemsCancelledByPlatform = new HashMap();
    for (var index = 0; index < items2Cancel.length; index++) {
        var cancelItem = cancelItems.get(items2Cancel[index]);
        if (!empty(cancelItem) && somPlatformCancellationHelpers.isCancelledByPlatform(cancelItem.get('reason'))) {
            itemsCancelledByPlatform.put(items2Cancel[index], cancelItem);
        }
    }
    if (!empty(itemsCancelledByPlatform)) {
        // update SFCC order status
        sfccOrders = new HashMap();
        sfccOrders.put(orderID, order);
        HookMgr.callHook('app.order.update.processStatusUpdateItemsByReason', 'processStatusUpdateItemsByReason', sfccOrders, itemsCancelledByPlatform);
        return true;
    }
    Logger.error('Couldn\'t retrieve order from SOM or already shipped and can\'t be cancelled');
    return false;
};

/**
 *   to cancel full order get all the product line items to cancel
 *   @param {Object} sfccOrder - sfcc order object
 *   @param {Object} reason - reason for cancellation
 *   @return {HashMap} - items to cancel
 */
function createCancelledItems(sfccOrder, reason) {
    var HashMap = require('dw/util/HashMap');
    var cancelledItems = new HashMap();

    var plis = sfccOrder.getProductLineItems().iterator();
    while (plis.hasNext()) {
        var pli = plis.next();
        if (!pli.isBonusProductLineItem()) {
            var itemCancelled = new HashMap();
            itemCancelled.put('quantity', pli.getQuantityValue());
            itemCancelled.put('reason', reason);
            cancelledItems.put(pli.productID, itemCancelled);
        }
    }
    return cancelledItems;
}

/**
*   cancel the whole order
*   @param {Object} order - sfcc order object
*   @param {Object} orderInput -  JSON of which items needs to be cancelled
*   @return {boolean} - return true if no API exceptions
*/
exports.processCancelOrder = function (order, orderInput) {
    var HashMap = require('dw/util/HashMap');
    var orderID = order.getOrderNo();
    var somOrder = orderHelpers.getOrderSummary([orderID]);
    var cancelReason = orderInput.c_cancelInfo.get('reason');
    var sfccOrders;
    // make sure the OMS order is not shipped
    if (somOrder != null && somOrder.length > 0 && somOrder[0].orderedStatusGroupItems != null && somOrder[0].orderedStatusGroupItems.length > 0) {
        if (cancelReason == null) {
            cancelReason = DEFAULT_CANCEL_REASON;
        }
        var cancelItems = createCancelledItems(order, cancelReason);
        var cancelData = getCancelOrderItems(somOrder[0].orderedStatusGroupItems[0], order, cancelItems);
        if (cancelData != null) {
            var somRes = som.cancelOrderItems(cancelData);
            if (somRes.ok && somRes.object.responseObj[0].isSuccess) {
                // update SFCC order status
                var orderNumbers = [];
                sfccOrders = new HashMap();
                orderNumbers.push(orderID);
                sfccOrders.put(orderID, order);
                HookMgr.callHook('app.order.update.processStatusUpdate', 'processStatusUpdate', JSON.stringify(orderNumbers), sfccOrders);
                return true;
            }
            Logger.error('Error cancelling order items in SOM');
        }
    }

    if (somPlatformCancellationHelpers.isCancelledByPlatform(cancelReason)) {
        // update SFCC order status
        sfccOrders = new HashMap();
        sfccOrders.put(orderID, order);
        HookMgr.callHook('app.order.update.processStatusUpdateByReason', 'processStatusUpdateByReason', sfccOrders, cancelReason);
        return true;
    }
    Logger.error('Couldn\'t retrieve order from SOM or already shipped and can\'t be cancelled');
    return false;
};
