'use strict';

/**
 * Get the quantity refunded for the given line item
 * @param {dw.order.ProductLineItem/dw.order.LineItemCtnr} pli - Item representing in the cart to refund
 * @returns {number} Quantity available to refund
 */
function getQuantityAvailableToRefund(pli) {
    var qtyAlreadyRefunded;
    if (Object.hasOwnProperty.call(pli, 'custom') && Object.hasOwnProperty.call(pli.custom, 'quantityReturned') && !empty(pli.custom.quantityReturned)) {
        qtyAlreadyRefunded = pli.custom.quantityReturned;
    } else {
        qtyAlreadyRefunded = 0;
    }

    var quantityAvailableToRefund = pli.quantity.value - qtyAlreadyRefunded;

    return quantityAvailableToRefund;
}

/**
 * Check if the item will be fully refunded after the transaction
 * @param {dw.order.ProductLineItem/dw.order.LineItemCtnr} pli - Item representing in the cart to refund
 * @param {number} quantityToRefund - Quantity to refund
 * @returns {boolean} Line Item Will Be Fully Refunded
 */
function isLineItemWillBeFullyRefunded(pli, quantityToRefund) {
    var qtyAlreadyReturned;
    if (Object.hasOwnProperty.call(pli, 'custom') && Object.hasOwnProperty.call(pli.custom, 'quantityReturned') && !empty(pli.custom.quantityReturned)) {
        qtyAlreadyReturned = pli.custom.quantityReturned;
    } else {
        qtyAlreadyReturned = 0;
    }

    var qtyWillBeReturned = qtyAlreadyReturned + quantityToRefund;

    if (pli.quantity.value === qtyWillBeReturned) {
        return true;
    }

    return false;
}

/**
 * Saves the refunded quantity in the line item
 * @param {dw.order.ProductLineItem/dw.order.LineItemCtnr} pli - Item representing in the cart to refund
 * @param {number} qtyToRefund - Quantity to refund
 */
function saveQuantityRefunded(pli, qtyToRefund) {
    var qtyAlreadyReturned;
    if (Object.hasOwnProperty.call(pli, 'custom') && Object.hasOwnProperty.call(pli.custom, 'quantityReturned') && !empty(pli.custom.quantityReturned)) {
        qtyAlreadyReturned = pli.custom.quantityReturned;
    } else {
        qtyAlreadyReturned = 0;
    }

    pli.custom.quantityReturned = qtyAlreadyReturned + qtyToRefund;
}

/**
 * Checks whether all line items in the order have been fully refunded
 * @param {dw.order.Order} order order
 * @returns {boolean} order fully fefunded
 */
function isOrderFullyRefunded(order) {
    var OrderUtilCode = require('*/cartridge/scripts/util/OrderUtilCode');

    var plis = order.getProductLineItems().iterator();
    var fullyRefunded = true;

    while (plis.hasNext() && fullyRefunded) {
        var pli = plis.next();
        var externalLineItemReturnStatusValue;
        if (Object.hasOwnProperty.call(pli, 'custom') && Object.hasOwnProperty.call(pli.custom, 'externalLineItemReturnStatus')) {
            externalLineItemReturnStatusValue = pli.custom.externalLineItemReturnStatus.value;
        }

        if (externalLineItemReturnStatusValue !== OrderUtilCode.EXTERNAL_LINE_ITEM_RETURN_STATUS.REFUNDED) {
            fullyRefunded = false;
        }
    }

    return fullyRefunded;
}

module.exports = {
    getQuantityAvailableToRefund: getQuantityAvailableToRefund,
    isLineItemWillBeFullyRefunded: isLineItemWillBeFullyRefunded,
    saveQuantityRefunded: saveQuantityRefunded,
    isOrderFullyRefunded: isOrderFullyRefunded
};
