'use strict';

var collections = require('*/cartridge/scripts/social/utils/fileCollections');
var File = require('dw/io/File');
var FileReader = require('dw/io/FileReader');
var FileWriter = require('dw/io/FileWriter');
var Logger = require('dw/system/Logger').getLogger('SocialBase', 'FileHelpers');

/**
 * Determines whether a string ends with the characters of suffix
 * @param {string} value string value
 * @param {string} suffix string suffix
 * @returns {boolean} validation
 */
function endsWith(value, suffix) {
    return value.indexOf(suffix, value.length - suffix.length) !== -1;
}

/**
 * Validate file name among accepted extensions
 *
 * @param {string} fileName The file name
 * @returns {boolean} Validation result
 */
function validateFileExtension(fileName) {
    var fileNameLower = fileName.toLowerCase();

    return endsWith(fileNameLower, '.csv')
        || endsWith(fileNameLower, '.tsv')
        || endsWith(fileNameLower, '.txt')
        || endsWith(fileNameLower, '.xml')
        || endsWith(fileNameLower, '.zip');
}

module.exports.addFileNameToTrackingFile = function (filePath, fileName) {
    var folderFile = new File(filePath);
    var file;
    var writer;

    try {
        if (!folderFile.exists() && !folderFile.mkdirs()) {
            Logger.error('Could not create folder: ' + filePath);
            return false;
        }

        file = new File(filePath + File.SEPARATOR + '.tracking');

        if (!file.exists() && !file.createNewFile()) {
            Logger.error('Could not create tracking file');
            return false;
        }

        writer = new FileWriter(file, true);
        writer.setLineSeparator('\n');
        writer.writeLine(fileName);
    } catch (e) {
        Logger.error('Error saving information into the tracking file: ' + e.message);
        return false;
    } finally {
        if (writer) {
            writer.close();
        }
    }
    return true;
};

module.exports.deleteTrackingFile = function (filePath) {
    var folderFile = new File(filePath);
    var file;

    try {
        if (!folderFile.exists()) {
            Logger.error('Could not find the folder: ' + filePath);
            return false;
        }

        file = new File(folderFile.fullPath + File.SEPARATOR + '.tracking');

        if (!file.exists()) {
            Logger.error('Could not find the tracking file');
            return false;
        }

        return file.remove();
    } catch (e) {
        Logger.error('Error deleting the tracking file: ' + e.message);
        return false;
    }
};

module.exports.getFilesToExport = function (filePath) {
    var folderFile = new File(filePath);
    var filesToExport = [];

    if (!folderFile.exists()) {
        return filesToExport;
    }

    if (folderFile.isDirectory()) {
        var trackingFile = new File(filePath + File.SEPARATOR + '.tracking');

        if (trackingFile.exists()) {
            var fileReader;
            var nextFile;

            try {
                fileReader = new FileReader(trackingFile);

                // eslint-disable-next-line no-cond-assign
                while ((nextFile = fileReader.readLine()) != null) {
                    var trackedFile = new File(filePath + File.SEPARATOR + nextFile);

                    if (trackedFile.exists() && validateFileExtension(nextFile)) {
                        filesToExport.push(trackedFile);
                    } else {
                        Logger.warn('There was an invalid entry inside .tracking file: ' + nextFile);
                    }
                }
            } catch (error) {
                Logger.warn(error.message);
            } finally {
                if (fileReader) {
                    fileReader.close();
                }
            }
        }
        if (!trackingFile.exists() || empty(filesToExport)) {
            // If there is no tracking file, we should get the most recent file inside folder
            var files = folderFile.listFiles();

            // Search for supported files
            files = collections.filter(files, function (f) {
                return !f.isDirectory() && validateFileExtension(f.getName());
            });

            if (files.length) {
                // Sort to get most recent
                files.sort(function (a, b) {
                    return a.lastModified() > b.lastModified() ? -1 : 1;
                });

                filesToExport.push(files.toArray().shift());
            }
        }
    } else if (validateFileExtension(folderFile.getName())) {
        return filesToExport.push(folderFile);
    }

    return filesToExport;
};

module.exports.endsWith = endsWith;
