'use strict';

var fileHelpers = require('*/cartridge/scripts/social/helpers/fileHelpers');
var File = require('dw/io/File');
var Logger = require('dw/system/Logger').getLogger('Instagram');

/**
 * @module scripts/jobs/orders
 */

/**
 * Export orders IDs to a text file separated by comma
 *
 * @param {string} basePath base path
 * @param {Long} lastModified last modified
 */
function generateExportOrdersFile(basePath, lastModified) {
    var Site = require('dw/system/Site');
    var FileWriter = require('dw/io/FileWriter');
    var OrderMgr = require('dw/order/OrderMgr');
    var dir = new File(basePath);
    var filename;
    var file;
    var writer;
    var entry;
    var timeStamp = new Date().getTime();
    var currentSite = Site.getCurrent();
    var iter;

    filename = 'orders-' + currentSite.ID + '-' + timeStamp + '.csv';

    try {
        if (!dir.exists() && !dir.mkdirs()) {
            Logger.error('Directory not exists.');
        }

        file = new File(dir, filename);

        if (!file.exists() && !file.createNewFile()) {
            Logger.error('Could not create file: ' + filename);
        }

        writer = new FileWriter(file);

        /** Get orders from Instagram Commerce channel type (value = 12) */
        if (lastModified === undefined) {
            iter = OrderMgr.searchOrders('channelType = {0}', null, 12);
        } else {
            iter = OrderMgr.searchOrders('channelType = {0} AND lastModified >= {1}', null, 12, new Date(lastModified));
        }

        writer.writeLine('SFCC_ORDER_ID, META_ORDER_ID');
        while (iter.hasNext()) {
            entry = iter.next();
            var orderNo = entry.orderNo;
            var externalOrderId = entry.custom && 'externalOrderId' in entry.custom && !empty(entry.custom.externalOrderId) ? entry.custom.externalOrderId : '';
            writer.writeLine(orderNo + ', ' + externalOrderId);
        }
    } catch (e) {
        Logger.error('Error saving information into file: ' + e.message);
    } finally {
        if (writer) {
            writer.close();
        }
    }
}

/**
 * Export orders to Meta
 *
 * @param {Object} parameters The job's parameters
 * @returns {dw/system/Status} The status of the job
 */
function exportOrders(parameters) {
    var Status = require('dw/system/Status');
    var StepHelper = require('int_instagram/cartridge/scripts/stepHelper');

    //  Is the current jobStep being skipped?  If so, exit early
    if (StepHelper.isDisabled(parameters)) {
        return new Status(Status.OK, 'SKIP', 'Step disabled, skip it...');
    }

    if (!empty(parameters.FileFolder)) {
        if (parameters.FileFolder[0] !== File.SEPARATOR) {
            parameters.FileFolder = File.SEPARATOR + parameters.FileFolder;
        }
        parameters.FileFolder = File.IMPEX + parameters.FileFolder;
    }

    var filesToExport = fileHelpers.getFilesToExport(parameters.FileFolder);
    var lastModified;

    if (!empty(filesToExport)) {
        lastModified = filesToExport[0].lastModified();
    }

    generateExportOrdersFile(parameters.FileFolder, lastModified);

    return new Status(Status.OK, 'OK', 'All exports succeed.');
}

module.exports.exportOrders = exportOrders;
