'use strict';

/**
 * All the nodes for Instagram BM extension
 * @module controllers/BM_Instagram
 */

var ISML = require('dw/template/ISML');
var Logger = require('dw/system/Logger').getLogger('Instagram', 'instagramModule');
var Resource = require('dw/web/Resource');
var TaxMgr = require('dw/order/TaxMgr');
var Transaction = require('dw/system/Transaction');
var URLUtils = require('dw/web/URLUtils');

var instagramService = require('int_instagram/cartridge/scripts/services/instagramService');
var instagramSfccService = require('int_instagram/cartridge/scripts/services/instagramSfccService');
var validationHelper = require('../scripts/utils/validationHelper');

var customObjectHelper = require('int_instagram/cartridge/scripts/customObjectHelper');

const MBE_STATE = {
    SETUP: 0,
    ONBOARD: 1,
    SHARE: 2,
    READY: 3
};

var breadcrumbs = [
    {
        htmlValue: Resource.msg('socialchannels.title', 'instagram', null),
        url: URLUtils.url('SiteNavigationBar-ShowMenuitemOverview', 'CurrentMenuItemId', 'social_channels_customadminmenuextension').toString()
    },
    {
        htmlValue: Resource.msg('instagram.title', 'instagram', null),
        url: URLUtils.url('BM_Instagram-Start').toString()
    }
];

/**
 * Get required cartridges to onboard
 * @param {string} hostname - hostname
 * @returns {string} returns required cartridges to onboard
 */
function getRequiredCartridgesToOnboard(hostname) {
    var Site = require('dw/system/Site');
    var currentSite = Site.getCurrent();
    var siteId = currentSite.ID;
    var requiredCartridgesToOnboard = [];

    try {
        if (!validationHelper.validateCartridge('GET', hostname, siteId, 'IntSocialCheckout-Loaded')) {
            requiredCartridgesToOnboard.push('int_social_checkout');
        }
        if (!validationHelper.validateCartridge('GET', hostname, siteId, 'IntOrderSom-Loaded') && !validationHelper.validateCartridge('GET', hostname, siteId, 'IntOrderNoOms-Loaded')) {
            requiredCartridgesToOnboard.push('int_order_som OR int_order_no_oms');
        }
    } catch (e) {
        Logger.warn('There was a problem trying to validade some cartridges.');
    }

    return requiredCartridgesToOnboard.toString();
}

/**
 * Accept terms for Instagram and save the flag in custom object
 */
function acceptTerms() {
    var instagramSettings = customObjectHelper.getSettings();

    Transaction.wrap(function () {
        instagramSettings.custom.acceptTerms = true;
    });

    response.redirect(URLUtils.https('BM_Instagram-Start', 'csrf_token', request.httpParameterMap.csrf_token.stringValue));
}

/**
 * Landing page for Instagram
 */
function start() {
    var connectionStatus = request.httpParameterMap.status.stringValue;
    var csrfToken = request.httpParameterMap.csrf_token.stringValue;

    var instagramSettings = customObjectHelper.getSettings();
    var instanceParams = customObjectHelper.getInstanceParams(instagramSettings);
    var onboardingStep = customObjectHelper.getOnboardingStep(instagramSettings);
    var taxPolicy = TaxMgr.getTaxationPolicy();
    var meta;
    var requiredCartridgesToOnboard;

    // Checking which tax policy is in use. If it is GROSS then renders the error page
    if (taxPolicy === TaxMgr.TAX_POLICY_GROSS) {
        ISML.renderTemplate('instagram/error', {
            Title: Resource.msg('instagram.error', 'instagram', null),
            Message: Resource.msg('instagram.integration.taxpolicy.error', 'instagram', null)
        });
        return;
    }

    // Check if the required cartridges are in the Site's cartridge path
    if (onboardingStep >= MBE_STATE.ONBOARD) {
        requiredCartridgesToOnboard = getRequiredCartridgesToOnboard(instagramSettings.custom.hostname);
    }

    // if its already onboarded it means that we have a token
    if (onboardingStep > MBE_STATE.ONBOARD) {
        var validateResponse = instagramService.validateAccessToken(instagramSettings);

        // if token could not be validated, disconnect MBE
        if (validateResponse.error || empty(validateResponse.result) || empty(validateResponse.result.id)) {
            Logger.error('An error occurred validating the current token.');
            connectionStatus = 'instagram.connection.expired';
            instagramService.disconnectMBE(instagramSettings);
            customObjectHelper.clearSettings(instagramSettings);
            onboardingStep = customObjectHelper.getOnboardingStep(instagramSettings);
        }
    }

    // if every step is completed, try to get the meta component
    if (onboardingStep === MBE_STATE.READY) {
        var metaResponse = instagramService.getCommerceExtensionUrl(instagramSettings);

        if (metaResponse.error || empty(metaResponse.result) || empty(metaResponse.result.commerce_extension)) {
            meta = {
                isConnected: false,
                status: 'instagram.meta.connection.error',
                uri: null
            };
        } else {
            meta = {
                isConnected: true,
                status: 'instagram.meta.connection.success',
                uri: metaResponse.result.commerce_extension.uri
            };
        }
    }

    // set form field values from custom object
    customObjectHelper.fillFormFromCustomObject(instagramSettings);

    // render the landing page so that the customer can authenticate through Instagram
    ISML.renderTemplate('instagram/start', {
        acceptTerms: instagramSettings.custom.acceptTerms,
        breadcrumbs: breadcrumbs,
        csrf_token: csrfToken,
        mbe: {
            appId: instanceParams.appId,
            externalBusinessId: instanceParams.externalBusinessId,
            onboardingStep: onboardingStep,
            splash: instanceParams.splashUrl,
            status: connectionStatus
        },
        meta: meta,
        page: {
            jobs: URLUtils.url('ViewApplication-BM', 'csrf_token', csrfToken) + '#/?job',
            scapiSettings: URLUtils.url('ViewCommerceApiSettings-Start', 'SelectedMenuItem', 'studio', 'CurrentMenuItemId', 'studio', 'csrf_token', csrfToken)
        },
        requiredCartridgesToOnboard: requiredCartridgesToOnboard
    });
}

/**
 * Update the required parameters before launching MBE flow
 */
function update() {
    var state = { ok: true, error: false };

    try {
        var body = JSON.parse(request.httpParameterMap.requestBodyAsString);
        var instagramSettings = customObjectHelper.getSettings();

        if (!empty(body.am_id) && !empty(body.am_secret)
                && !empty(body.feed_upload_id) && !empty(body.hostname)
                && !empty(body.slas_id) && !empty(body.slas_secret)
                && !empty(body.short_code) && !empty(body.org_id)) {
            customObjectHelper.clearSettings(instagramSettings);

            var valAMCredentials = instagramSfccService.validateAMCredentials(body.am_id, body.am_secret, body.org_id);
            var valTSOBCredentials = instagramSfccService.validatePrivateTSOBClient(body.slas_id, body.slas_secret, body.short_code, body.org_id);
            var valHostname = validationHelper.validateHostname(body.hostname);

            Transaction.wrap(function () {
                if (valAMCredentials && !valAMCredentials.error) {
                    instagramSettings.custom.amClientId = body.am_id;
                    instagramSettings.custom.amClientSecret = body.am_secret;
                } else {
                    instagramSettings.custom.amClientId = null;
                    instagramSettings.custom.amClientSecret = null;
                }

                instagramSettings.custom.feedUploadId = body.feed_upload_id;
                instagramSettings.custom.hostname = valHostname ? body.hostname : null;
                instagramSettings.custom.shopperClientId = valTSOBCredentials ? body.slas_id : null;
                instagramSettings.custom.shopperClientSecret = valTSOBCredentials ? body.slas_secret : null;
                instagramSettings.custom.shortCode = body.short_code;
                instagramSettings.custom.orgId = body.org_id;
            });

            if (valAMCredentials && valAMCredentials.error && !valTSOBCredentials) {
                throw new Error(Resource.msgf('instagram.invalid.credentials.all', 'instagram', null, valAMCredentials.errorDescription));
            } else if (valAMCredentials && valAMCredentials.error) {
                throw new Error(Resource.msgf('instagram.invalid.credentials.am', 'instagram', null, valAMCredentials.errorDescription));
            } else if (!valTSOBCredentials) {
                throw new Error(Resource.msg('instagram.invalid.credentials.slas', 'instagram', null));
            } else if (!valHostname) {
                throw new Error(Resource.msg('instagram.invalid.hostname', 'instagram', null));
            }
        } else {
            throw new Error(Resource.msg('instagram.missing.required.parameters', 'instagram', null));
        }
    } catch (error) {
        state = { ok: false, error: true, message: error.message };
        response.setStatus(400);
    }

    response.setContentType('application/json');
    response.writer.print(JSON.stringify(state));
}

/**
 * Callback for MBE flow
 */
function callback() {
    var instagramSettings = customObjectHelper.getSettings();
    var state = { ok: true, error: false, redirect_url: URLUtils.https('BM_Instagram-Start', 'csrf_token', request.httpParameterMap.csrf_token.stringValue, 'status', 'instagram.connection.success').toString() };

    try {
        var body = JSON.parse(request.httpParameterMap.requestBodyAsString);

        if (body && body.message && body.message.success && !empty(body.message.access_token)) {
            var accessToken = body.message.access_token;
            var appId = body.app_id;
            var businessManagerId = body.message.business_manager_id;
            var commercePartnerIntegrationId = body.message.commerce_partner_integration_id;
            var externalBusinessId = body.external_business_id;

            Transaction.wrap(function () {
                instagramSettings.custom.appId = appId;
                instagramSettings.custom.businessManagerId = businessManagerId;
                instagramSettings.custom.commercePartnerIntegrationId = commercePartnerIntegrationId;
                instagramSettings.custom.externalBusinessId = externalBusinessId;
                instagramSettings.custom.userToken = accessToken;
            });

            var postOnboarding = instagramService.postOnboardingData(instagramSettings);

            if (postOnboarding && !postOnboarding.error && postOnboarding.result && postOnboarding.result.success) {
                Transaction.wrap(function () {
                    instagramSettings.custom.metaConnected = true;
                });
            } else {
                Logger.error('An error occurred after the onboarding process. Post onboarding data was not shared: {0}', JSON.stringify(postOnboarding));
                state = { ok: false, error: true, redirect_url: URLUtils.https('BM_Instagram-Start', 'csrf_token', request.httpParameterMap.csrf_token.stringValue, 'status', 'instagram.share.error').toString(), reload: true };
                response.setStatus(400);
            }
        } else {
            Logger.error('An error occurred finishing the onboarding process. Message status or token was not received: {0}', JSON.stringify(body ? body.message : null));
            throw new Error('Error');
        }
    } catch (error) {
        customObjectHelper.clearSettings(instagramSettings);
        state = { ok: false, error: true, message: error.message, redirect_url: URLUtils.https('BM_Instagram-Start', 'csrf_token', request.httpParameterMap.csrf_token.stringValue, 'status', 'instagram.connection.invalid').toString() };
        response.setStatus(400);
    }

    response.setContentType('application/json');
    response.writer.print(JSON.stringify(state));
}

/**
 * Share MBE post onboarding data
 */
function share() {
    var instagramSettings = customObjectHelper.getSettings();
    var state = { ok: true, error: false, redirect_url: URLUtils.https('BM_Instagram-Start', 'csrf_token', request.httpParameterMap.csrf_token.stringValue, 'status', 'instagram.connection.success').toString() };

    try {
        var postOnboarding = instagramService.postOnboardingData(instagramSettings);

        if (postOnboarding && !postOnboarding.error && postOnboarding.result && postOnboarding.result.success) {
            Transaction.wrap(function () {
                instagramSettings.custom.metaConnected = true;
            });
        } else {
            Logger.error('An error occurred trying to share the post onboarding data: {0}', JSON.stringify(postOnboarding));
            throw new Error(Resource.msg('instagram.share.error', 'instagram', null));
        }
    } catch (error) {
        state = { ok: false, error: true, message: error.message, redirect_url: URLUtils.https('BM_Instagram-Start', 'csrf_token', request.httpParameterMap.csrf_token.stringValue, 'status', 'instagram.share.error').toString() };
        response.setStatus(400);
    }

    response.setContentType('application/json');
    response.writer.print(JSON.stringify(state));
}

/**
 * Disconnect MBE and clear system user access token
 */
function disconnect() {
    var instagramSettings = customObjectHelper.getSettings();

    instagramService.disconnectMBE(instagramSettings);
    customObjectHelper.clearSettings(instagramSettings);
    response.redirect(URLUtils.https('BM_Instagram-Start', 'csrf_token', request.httpParameterMap.csrf_token.stringValue, 'status', 'instagram.connection.disconnected').toString());
}

/**
 * Endpoints
 */
module.exports.AcceptTerms = acceptTerms;
module.exports.AcceptTerms.public = true;
module.exports.Start = start;
module.exports.Start.public = true;
module.exports.Update = update;
module.exports.Update.public = true;
module.exports.Callback = callback;
module.exports.Callback.public = true;
module.exports.Share = share;
module.exports.Share.public = true;
module.exports.Disconnect = disconnect;
module.exports.Disconnect.public = true;
